let currentDataType = 'aio'; // 'aio', 'aim', or 'cross'
let globalSubTab = 'aio'; // 'aio' or 'aim' — for the Global Data sub-tab toggle
const Y_STYLE = 'color:#00895e;font-weight:600;';
const N_STYLE = 'color:#ccc;';
function escapeHtml(unsafe) {
    if (typeof unsafe !== 'string') return '';
    return unsafe
        .replace(/&/g, "&amp;")
        .replace(/</g, "&lt;")
        .replace(/>/g, "&gt;")
        .replace(/"/g, "&quot;")
        .replace(/'/g, "&#039;");
}
function faviconImg(domain) {
    if (!domain || domain === '—') return '';
    return `<img src="https://www.google.com/s2/favicons?domain=${encodeURIComponent(domain)}&sz=16" width="16" height="16" style="vertical-align:middle;margin-right:4px;" onerror="this.style.display='none'">`;
}
function extractDomain(url) {
    try {
        const urlObj = new URL(url);
        return stripTranslateGoog(urlObj.hostname.replace('www.', ''));
    } catch (e) {
        const match = url.match(/\/\/(?:www\.)?([^\/]+)/);
        return match ? match[1] : url;
    }
}
function stripTranslateGoog(hostname) {
    if (!hostname.endsWith('.translate.goog')) return hostname;
    return hostname.replace('.translate.goog', '').replace(/--/g, '.');
}
function cleanSourceLabel(label) {
    if (!label) return '';
    return label.replace(/^"/, '').replace(/"$/, '').replace(/^Source:\s*/i, '');
}
function isProductLink(href) {
    if (!href) return false;
    return href.includes('q=product') || href.includes('ibp=oshop');
}
function cleanGoogleSearchUrl(url) {
    if (!url) return url;
    try {
        const u = new URL(url);
        if (u.hostname.includes('google.com') && u.pathname === '/search' && u.searchParams.has('q')) {
            if (u.searchParams.get('q') === 'product' || u.searchParams.has('ibp')) return url;
            return `${u.origin}/search?q=${u.searchParams.get('q')}`;
        }
    } catch (e) {}
    return url;
}
function normalizeUrlForComparison(url, domainLevel) {
    if (!url) return '';
    try {
        const urlObj = new URL(url);
        const hostname = stripTranslateGoog(urlObj.hostname.replace('www.', ''));
        if (domainLevel) return hostname;
        let norm = hostname + urlObj.pathname.replace(/\/$/, '');
        if (urlObj.search) norm += urlObj.search;
        return norm;
    } catch (e) {
        return url.replace(/^https?:\/\/(www\.)?/, '').replace(/#.*$/, '').replace(/\/$/, '');
    }
}
function getTimeAgo(date) {
    if (!date) return 'Unknown date';
    try {
        const seconds = Math.floor((new Date() - date) / 1000);
        if (seconds < 60) return 'Just now';
        if (seconds < 3600) return `${Math.floor(seconds / 60)} min ago`;
        if (seconds < 86400) return `${Math.floor(seconds / 3600)} h ago`;
        if (seconds < 604800) return `${Math.floor(seconds / 86400)} d ago`;
        return date.toLocaleDateString('en-US');
    } catch (error) {
        return 'Invalid date';
    }
}
function isExcludedUrl(url) {
    if (!url) return true;
    if (/^https?:\/\/(www\.)?google\.com\/?$/.test(url)) return true;
    if (url.includes('w3.org/')) return true;
    if (url.includes('policies.google.com')) return true;
    if (url.includes('googleusercontent.com')) return true;
    return false;
}
function formatTextFragment(rawFragment) {
    if (!rawFragment) return '';
    const fragments = rawFragment.split('&text=').map(f => f.trim()).filter(f => f);
    if (fragments.length <= 1) return escapeHtml(rawFragment);
    return '<ul style="margin:0;padding-left:18px;font-size:11px;">' +
        fragments.map(f => `<li style="margin-bottom:2px;">${escapeHtml(f)}</li>`).join('') +
        '</ul>';
}
async function navigateToQuery(queryText) {
    const allData = await new Promise(r => chrome.storage.local.get(null, r));
    const queries = processStorageData(allData, 'all');
    const match = queries.find(q => q.query === queryText);
    if (!match) return;
    const hasAIO = match.captures.some(c => c.captureType === 'aio');
    const hasAIM = match.captures.some(c => c.captureType === 'aim');
    if (hasAIO && !hasAIM && currentDataType === 'aim') await switchDataType('aio');
    else if (hasAIM && !hasAIO && currentDataType === 'aio') await switchDataType('aim');
    const typedQueries = processStorageData(allData, currentDataType);
    const idx = typedQueries.findIndex(q => q.query === queryText);
    const targetQuery = idx >= 0 ? typedQueries[idx] : match;
    document.getElementById('allQueriesContainer').style.display = 'none';
    document.getElementById('allDataAttridContainer').style.display = 'none';
    document.getElementById('noDataContainer').style.display = 'none';
    document.getElementById('dataContainer').style.display = 'block';
    document.getElementById('showAllBtn').style.display = 'block';
    document.getElementById('showDataAttridBtn').style.display = 'block';
    document.getElementById('showSingleBtn').style.display = 'none';
    const selectEl = document.getElementById('querySelect');
    selectEl.style.display = 'block';
    if (idx >= 0) selectEl.value = idx.toString();
    await renderQueryDetail(targetQuery);
}
function renderQueryLinks(queries) {
    return queries.map(q => `<a href="#" class="query-nav-link" data-query="${escapeHtml(q)}" style="color:var(--primary);text-decoration:none;cursor:pointer;" title="${escapeHtml(q)}">${escapeHtml(q)}</a>`).join(', ');
}
function renderSingleQueryLink(q) {
    return `<a href="#" class="query-nav-link" data-query="${escapeHtml(q)}" style="color:var(--primary);text-decoration:none;cursor:pointer;" title="${escapeHtml(q)}">${escapeHtml(q)}</a>`;
}
function wireQueryNavLinks(container) {
    container.querySelectorAll('.query-nav-link').forEach(link => {
        link.addEventListener('click', async (e) => {
            e.preventDefault();
            await navigateToQuery(link.dataset.query);
        });
    });
}
async function getAllStorageData(filterType) {
    const type = filterType || currentDataType;
    return new Promise((resolve) => {
        chrome.storage.local.get(null, (data) => {
            const captures = {};
            const prefixes = (type === 'cross' || type === 'all') ? ['aio_', 'aim_'] : [type === 'aim' ? 'aim_' : 'aio_'];
            for (const [key, value] of Object.entries(data || {})) {
                if (prefixes.some(p => key.startsWith(p)) && typeof value === 'object' && value.searchQuery) {
                    captures[key] = value;
                }
            }
            resolve(captures);
        });
    });
}
function processStorageData(data, filterType) {
    const type = filterType || currentDataType;
    const queries = {};
    const prefixes = (type === 'cross' || type === 'all') ? ['aio_', 'aim_'] : [type === 'aim' ? 'aim_' : 'aio_'];
    for (const [key, value] of Object.entries(data)) {
        if (!prefixes.some(p => key.startsWith(p)) || typeof value !== 'object' || !value.searchQuery) continue;
        const query = value.searchQuery || 'unknown query';
        const captureType = key.startsWith('aim_') ? 'aim' : 'aio';
        if (!queries[query]) {
            queries[query] = {
                query: query,
                captures: []
            };
        }
        const captureObj = {
            key: key,
            captureType: captureType,
            searchQuery: value.searchQuery,
            citations: value.citations || [],
            generatedText: value.generatedText || '',
            entityLinks: value.entityLinks || [],
            externalLinks: value.externalLinks || [],
            magiInnerHTML: value.magiInnerHTML || [],
            timestamp: value.timestamp,
            pageUrl: value.pageUrl || ''
        };
        captureObj.imageCarouselSources = value.imageCarouselSources || [];
        if (captureType === 'aio') {
            captureObj.aioStatus = value.aioStatus || 'absent';
            captureObj.fanOutQuery = value.fanOutQuery || null;
            captureObj.followUpQuestions = value.followUpQuestions || null;
            captureObj.additionalTextFragments = value.additionalTextFragments || [];
            captureObj.sidebarCards = value.sidebarCards || [];
            captureObj.citationButtonUUIDs = value.citationButtonUUIDs || [];
            captureObj.organicUrls = value.organicUrls || [];
            captureObj.textFragmentLinks = value.textFragmentLinks || [];
        } else {
            captureObj.aimStatus = value.aimStatus || 'absent';
            captureObj.sidebarSources = value.sidebarSources || [];
            captureObj.placeEntities = value.placeEntities || [];
            captureObj.kgEntities = value.kgEntities || [];
            captureObj.ffdbKGIds = value.ffdbKGIds || [];
            captureObj.textFragmentLinks = value.textFragmentLinks || [];
        }
        queries[query].captures.push(captureObj);
    }
    for (const query of Object.values(queries)) {
        query.captures.sort((a, b) => {
            const timeA = a.timestamp ? new Date(a.timestamp).getTime() : 0;
            const timeB = b.timestamp ? new Date(b.timestamp).getTime() : 0;
            return timeB - timeA;
        });
    }
    let result = Object.values(queries);
    if (type === 'cross') {
        result = result.filter(q => {
            const hasAIO = q.captures.some(c => c.captureType === 'aio');
            const hasAIM = q.captures.some(c => c.captureType === 'aim');
            return hasAIO && hasAIM;
        });
    }
    return result.sort((a, b) => {
        const aTime = Math.max(...(a.captures || []).map(c => c.timestamp ? new Date(c.timestamp).getTime() : 0));
        const bTime = Math.max(...(b.captures || []).map(c => c.timestamp ? new Date(c.timestamp).getTime() : 0));
        return bTime - aTime;
    });
}
async function loadQueries() {
    try {
        const data = await getAllStorageData();
        const queries = processStorageData(data);
        const select = document.getElementById('querySelect');
        select.innerHTML = '<option value="">Select a query...</option>';
        queries.forEach((query, index) => {
            const option = document.createElement('option');
            option.value = index;
            const citCount = query.captures.reduce((sum, c) => sum + (c.citations || []).length, 0);
            const sidebarCount = query.captures.reduce((sum, c) => {
                return sum + (c.captureType === 'aim' ? (c.sidebarSources || []).length : (c.sidebarCards || []).length);
            }, 0);
            option.textContent = `${query.query} (${citCount} citations · ${sidebarCount} in sidebar)`;
            select.appendChild(option);
        });
        return queries;
    } catch (error) {
        const select = document.getElementById('querySelect');
        select.innerHTML = '<option value="">Error loading...</option>';
        throw error;
    }
}
function showLoading(show) {
    document.getElementById('loadingContainer').style.display = show ? 'block' : 'none';
    document.getElementById('noDataContainer').style.display = !show && !document.getElementById('dataContainer').style.display.includes('block') ? 'block' : 'none';
}
function showData(show) {
    document.getElementById('dataContainer').style.display = show ? 'block' : 'none';
    document.getElementById('noDataContainer').style.display = !show ? 'block' : 'none';
}
function buildContextChips(capture, label) {
    const isAIM = capture.captureType === 'aim';
    let html = `<div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(180px, 1fr)); gap: 12px;">`;
    if (label) {
        html += `
            <div class="info-chip" style="background: var(--border-light); padding: 8px 12px; border-radius: 4px;">
                <span style="color: var(--text-secondary); font-size: 11px; text-transform: uppercase; letter-spacing: 0.5px;">Type</span>
                <div style="color: var(--text-primary); font-weight: 600; margin-top: 2px;">${escapeHtml(label)}</div>
            </div>
        `;
    }
    if (capture.timestamp) {
        const date = new Date(capture.timestamp);
        html += `
            <div class="info-chip" style="background: var(--border-light); padding: 8px 12px; border-radius: 4px;">
                <span style="color: var(--text-secondary); font-size: 11px; text-transform: uppercase; letter-spacing: 0.5px;">Date</span>
                <div style="color: var(--text-primary); font-weight: 500; margin-top: 2px;">${date.toLocaleDateString('en-US', { day: 'numeric', month: 'short', year: 'numeric' })} ${date.toLocaleTimeString('en-US', { hour: '2-digit', minute: '2-digit' })}</div>
            </div>
        `;
    }
    const statusValue = isAIM ? (capture.aimStatus || 'absent') : (capture.aioStatus || 'absent');
    html += `
        <div class="info-chip" style="background: var(--border-light); padding: 8px 12px; border-radius: 4px;">
            <span style="color: var(--text-secondary); font-size: 11px; text-transform: uppercase; letter-spacing: 0.5px;">Status</span>
            <div style="color: var(--text-primary); font-weight: 500; margin-top: 2px;">${escapeHtml(statusValue === 'magi_only' ? 'Grounding only' : statusValue)}</div>
        </div>
    `;
    const citCount = (capture.citations || []).length;
    const sidebarCount = isAIM ? (capture.sidebarSources || []).length : (capture.sidebarCards || []).length;
    html += `
        <div class="info-chip" style="background: var(--border-light); padding: 8px 12px; border-radius: 4px;">
            <span style="color: var(--text-secondary); font-size: 11px; text-transform: uppercase; letter-spacing: 0.5px;">Citations</span>
            <div style="color: var(--text-primary); font-weight: 500; margin-top: 2px;">${citCount} pool · ${sidebarCount} sidebar</div>
        </div>
    `;
    if (isAIM) {
        const entityCount = (capture.entityLinks || []).length;
        const inTextCount = (capture.externalLinks || []).length;
        html += `
            <div class="info-chip" style="background: var(--border-light); padding: 8px 12px; border-radius: 4px;">
                <span style="color: var(--text-secondary); font-size: 11px; text-transform: uppercase; letter-spacing: 0.5px;">Entities · In-text Links</span>
                <div style="color: var(--text-primary); font-weight: 500; margin-top: 2px;">${entityCount} · ${inTextCount}</div>
            </div>
        `;
    } else {
        const magiCount = (capture.magiInnerHTML || []).filter(m => !isExcludedUrl(m.url || '')).length;
        if (magiCount > 0) {
            html += `
                <div class="info-chip" style="background: var(--border-light); padding: 8px 12px; border-radius: 4px;">
                    <span style="color: var(--text-secondary); font-size: 11px; text-transform: uppercase; letter-spacing: 0.5px;">Grounding URLs</span>
                    <div style="color: var(--text-primary); font-weight: 500; margin-top: 2px;">${magiCount}</div>
                </div>
            `;
        }
    }
    html += '</div>';
    return html;
}
async function renderQueryDetail(query) {
    const contextCardContainer = document.getElementById('contextCard');
    const snapshotAIContentEl = document.getElementById('snapshotAIContent');
    const snapshotAISectionEl = document.getElementById('snapshotAISection');
    const additionalSections = document.getElementById('additionalSections');
    if (contextCardContainer) contextCardContainer.innerHTML = '';
    if (snapshotAIContentEl) snapshotAIContentEl.innerHTML = '';
    if (snapshotAISectionEl) snapshotAISectionEl.style.display = 'none';
    if (additionalSections) additionalSections.innerHTML = '';
    const vennSection = document.getElementById('vennSection');
    if (vennSection) { vennSection.style.display = 'none'; document.getElementById('vennContent').innerHTML = ''; }
    const crossSection = document.getElementById('crossComparisonSection');
    if (crossSection) crossSection.style.display = 'none';
    if (!query.captures || query.captures.length === 0) {
        return;
    }
    const capture = query.captures[0]; // Most recent capture
    const dataContainer = document.getElementById('dataContainer');
    if (dataContainer && vennSection && snapshotAISectionEl) {
        dataContainer.insertBefore(snapshotAISectionEl, vennSection);
    }
    let contextHtml = `
        <div class="card" style="background: var(--card); margin-bottom: 16px;">
            <div style="margin-bottom: 12px;">
                <div style="font-family: var(--font-heading); font-size: 24px; font-weight: 700; color: var(--text-primary); margin-bottom: 16px;">
                    ${escapeHtml(query.query)}
                </div>
            </div>
            ${buildContextChips(capture)}
        </div>
    `;
    if (contextCardContainer) contextCardContainer.innerHTML = contextHtml;
    if (snapshotAIContentEl && snapshotAISectionEl) {
        let snapshotHtml = '';
        let hasContent = false;
        {
            const entityLinks = capture.entityLinks || [];
            const kgEntities = (capture.captureType === 'aim') ? (capture.kgEntities || []) : [];
            const ffdbIds = (capture.captureType === 'aim') ? (capture.ffdbKGIds || []) : [];
            const placeMids = new Set((capture.placeEntities || []).map(p => p.mid).filter(Boolean));
            const mergedEntities = [];
            const seenNames = new Set();
            entityLinks.forEach(link => {
                if (isProductLink(link.href || '')) return;
                const name = (link.text || '').trim().toLowerCase();
                let linkText = link.text || '?';
                if ((!linkText || linkText === '?') && link.href) {
                    try { const u = new URL(link.href); const qp = u.searchParams.get('q'); if (qp) linkText = qp.replace(/\+/g, ' '); } catch(e) {}
                }
                const entry = { text: linkText, href: link.href || '', kgId: null };
                const kgMatch = kgEntities.find(k => k.name && k.name.trim().toLowerCase() === name);
                if (kgMatch && !placeMids.has(kgMatch.kgmid)) {
                    entry.kgId = kgMatch.kgmid;
                    entry.href = entry.href || kgMatch.href || '';
                }
                if (!entry.kgId) {
                    const ffdbMatch = ffdbIds.find(f => f.associatedName && f.associatedName.trim().toLowerCase() === name);
                    if (ffdbMatch && !placeMids.has(ffdbMatch.kgId)) {
                        entry.kgId = ffdbMatch.kgId;
                    }
                }
                mergedEntities.push(entry);
                seenNames.add(name);
            });
            kgEntities.forEach(e => {
                if (placeMids.has(e.kgmid)) return;
                const name = (e.name || '').trim().toLowerCase();
                if (seenNames.has(name)) return;
                seenNames.add(name);
                let kgText = e.name || '?';
                if ((!kgText || kgText === '?') && e.href) {
                    try { const u = new URL(e.href); const qp = u.searchParams.get('q'); if (qp) kgText = qp.replace(/\+/g, ' '); } catch(ex) {}
                }
                mergedEntities.push({ text: kgText, href: e.href || '', kgId: e.kgmid || null });
            });
            ffdbIds.forEach(e => {
                if (placeMids.has(e.kgId)) return;
                const name = (e.associatedName || '').trim().toLowerCase();
                if (!name || seenNames.has(name)) return;
                seenNames.add(name);
                mergedEntities.push({ text: e.associatedName || '?', href: '', kgId: e.kgId || null });
            });
            if (mergedEntities.length > 0) {
                hasContent = true;
                snapshotHtml += `<div style="margin-bottom:12px;"><strong style="font-size:12px;color:var(--text-secondary);">Entities (${mergedEntities.length})</strong></div>`;
                snapshotHtml += '<div style="display:flex;flex-wrap:wrap;gap:6px;margin-bottom:16px;">';
                mergedEntities.forEach(ent => {
                    const kgBadge = ent.kgId ? ` <span style="font-size:9px;color:var(--text-secondary);font-family:var(--font-mono);" title="${escapeHtml(ent.kgId)}">${escapeHtml(ent.kgId)}</span>` : '';
                    snapshotHtml += `<a href="${escapeHtml(ent.href)}" target="_blank" style="padding:4px 10px;background:var(--border-light);border-radius:16px;font-size:12px;color:var(--primary);text-decoration:none;border:1px solid var(--border);">${escapeHtml(ent.text)}${kgBadge}</a>`;
                });
                snapshotHtml += '</div>';
            }
        }
        {
            const productLinks = (capture.entityLinks || []).filter(link => isProductLink(link.href || ''));
            if (productLinks.length > 0) {
                hasContent = true;
                const typeLabel = capture.captureType === 'aim' ? 'AIM' : 'AIO';
                const sidebarProdCount = productLinks.filter(l => l.inSidebar).length;
                snapshotHtml += `<div style="margin-bottom:8px;"><strong style="font-size:12px;color:var(--text-secondary);">${typeLabel} Products (${productLinks.length})</strong>${sidebarProdCount > 0 ? ` <span style="font-size:11px;color:var(--text-secondary);">· ${sidebarProdCount} in sidebar</span>` : ''}</div>`;
                snapshotHtml += '<div style="display:flex;flex-wrap:wrap;gap:6px;margin-bottom:16px;">';
                productLinks.forEach(link => {
                    const href = cleanGoogleSearchUrl(link.href || '');
                    const sidebarBorder = link.inSidebar ? 'border:2px solid #e65100;' : 'border:1px solid #ffcc80;';
                    snapshotHtml += `<a href="${escapeHtml(href)}" target="_blank" style="padding:4px 10px;background:#fff3e0;border-radius:16px;font-size:12px;color:#e65100;text-decoration:none;${sidebarBorder}">${escapeHtml(link.text || '?')}${link.inSidebar ? ' <span style="font-size:10px;">&#x2713;</span>' : ''}</a>`;
                });
                snapshotHtml += '</div>';
            }
        }
        if (capture.externalLinks && capture.externalLinks.length > 0) {
            hasContent = true;
            snapshotHtml += `<div style="margin-bottom:8px;"><strong style="font-size:12px;color:var(--text-secondary);">In-text links (${capture.externalLinks.length})</strong></div>`;
            snapshotHtml += '<div style="display:flex;flex-direction:column;gap:4px;margin-bottom:16px;">';
            capture.externalLinks.forEach(link => {
                snapshotHtml += `<div style="font-size:12px;">${faviconImg(link.domain)}<a href="${escapeHtml(link.href || '')}" target="_blank" style="color:var(--primary);text-decoration:none;">${escapeHtml(link.domain || link.href || '')}</a> <span style="color:var(--text-secondary);">— ${escapeHtml(link.text || '')}</span></div>`;
            });
            snapshotHtml += '</div>';
        }
        if (capture.sidebarCards && capture.sidebarCards.length > 0) {
            hasContent = true;
            snapshotHtml += `<div style="margin-bottom:8px;"><strong style="font-size:12px;color:var(--text-secondary);">Sidebar citations (${capture.sidebarCards.length})</strong></div>`;
            snapshotHtml += '<div style="display:flex;flex-direction:column;gap:4px;margin-bottom:16px;">';
            capture.sidebarCards.forEach((card, i) => {
                const domain = card.domain || extractDomain(card.url || '') || '?';
                const title = card.title || '';
                snapshotHtml += `<div style="font-size:12px;">${faviconImg(domain)}<a href="${escapeHtml(card.url || '')}" target="_blank" style="color:var(--primary);text-decoration:none;">${escapeHtml(domain)}</a>${title ? ' <span style="color:var(--text-secondary);">' + escapeHtml(title) + '</span>' : ''}</div>`;
            });
            snapshotHtml += '</div>';
        }
        {
            const allCitations = [
                ...(capture.citations || []),
                ...(capture.sidebarCards || [])
            ];
            const seenFragUrls = new Set();
            const fragEntries = [];
            allCitations.forEach(c => {
                const url = c.url || '';
                if (!url) return;
                let frag = c.textFragment || '';
                if (!frag && url.includes('#:~:text=')) {
                    const fm = url.match(/#:~:text=(.+)/);
                    if (fm) { try { frag = decodeURIComponent(fm[1]); } catch (e) { frag = fm[1]; } }
                }
                if (!frag && !c.sectionHeading) return;
                const normUrl = url.replace(/#.*$/, '');
                if (seenFragUrls.has(normUrl)) return;
                seenFragUrls.add(normUrl);
                let sectionHeading = c.sectionHeading || '';
                if (!sectionHeading && frag.startsWith('[')) {
                    const bracketEnd = frag.indexOf('] ');
                    if (bracketEnd > 0 && bracketEnd < 150) {
                        sectionHeading = frag.substring(1, bracketEnd);
                        frag = frag.substring(bracketEnd + 2);
                    } else if (frag.endsWith(']') && frag.indexOf(']') === frag.length - 1 && frag.length < 150) {
                        sectionHeading = frag.substring(1, frag.length - 1);
                        frag = '';
                    }
                }
                if (!frag && !sectionHeading) return;
                fragEntries.push({ url, sourceDomain: c.sourceDomain || c.domain || '', textFragment: frag, publishedDate: c.publishedDate || '', sectionHeading });
            });
            if (fragEntries.length > 0) {
                hasContent = true;
                snapshotHtml += `<div style="margin-top:8px;">
                    <div style="font-size:11px;text-transform:uppercase;letter-spacing:0.5px;color:var(--text-secondary);font-weight:600;margin-bottom:8px;">Text Fragments</div>
                    <table class="magi-sources-table">
                        <thead><tr>
                            <th style="width:220px;">Source</th>
                            <th style="width:180px;">Section</th>
                            <th>Fragment</th>
                        </tr></thead><tbody>`;
                fragEntries.forEach(cit => {
                    const domain = cit.sourceDomain || extractDomain(cit.url || '') || '—';
                    snapshotHtml += `<tr>
                        <td style="font-size:12px;white-space:nowrap;"><a href="${escapeHtml(cit.url)}" target="_blank" style="color:var(--primary);text-decoration:none;">${faviconImg(domain)}${escapeHtml(domain)}</a></td>
                        <td style="font-size:12px;color:var(--text-secondary);font-style:italic;">${escapeHtml(cit.sectionHeading || '')}</td>
                        <td style="font-size:12px;color:var(--text-secondary);">${formatTextFragment(cit.textFragment)}</td>
                    </tr>`;
                });
                snapshotHtml += '</tbody></table></div>';
            }
        }
        if (capture.imageCarouselSources && capture.imageCarouselSources.length > 0) {
            hasContent = true;
            snapshotHtml += `<div style="margin-top:8px;">
                <div style="font-size:11px;text-transform:uppercase;letter-spacing:0.5px;color:var(--text-secondary);font-weight:600;margin-bottom:8px;">Image Sources (${capture.imageCarouselSources.length})</div>
                <table class="magi-sources-table">
                    <thead><tr>
                        <th style="width:30px;">#</th>
                        <th>URL</th>
                        <th style="width:200px;">Title</th>
                    </tr></thead><tbody>`;
            capture.imageCarouselSources.forEach((src, i) => {
                const domain = src.domain || extractDomain(src.url || '') || '—';
                snapshotHtml += `<tr>
                    <td style="text-align:center;color:var(--text-secondary);font-size:11px;">${i + 1}</td>
                    <td style="font-size:11px;word-break:break-all;">${faviconImg(domain)}<a href="${escapeHtml(src.url || '')}" target="_blank" style="color:var(--primary);text-decoration:none;">${escapeHtml(src.url || '')}</a></td>
                    <td style="font-size:12px;">${escapeHtml(src.title || '')}</td>
                </tr>`;
            });
            snapshotHtml += '</tbody></table></div>';
        }
        if (capture.generatedText) {
            hasContent = true;
            const formattedText = formatAIOGeneratedText(capture.generatedText);
            snapshotHtml += `<div class="aio-generated-text" style="margin-top:16px;">
                <div class="aio-text-label">Generated Response Structure</div>
                <div class="aio-text-body">${formattedText}</div>
                <button class="aio-text-toggle">Show more ▼</button>
            </div>`;
        }
        if (hasContent) {
            snapshotAISectionEl.style.display = 'block';
            snapshotAIContentEl.innerHTML = snapshotHtml;
            const snapshotTitleEl = snapshotAISectionEl.querySelector('.section-title');
            if (snapshotTitleEl) {
                const isAIM = capture.captureType === 'aim';
                const infoTip = snapshotTitleEl.querySelector('.info-icon');
                const infoHtml = infoTip ? infoTip.outerHTML : '';
                snapshotTitleEl.innerHTML = `<span class="section-icon">✨</span> ${isAIM ? 'AIM' : 'AIO'} Answer${infoHtml}`;
            }
            if (capture.generatedText) {
                const toggleBtn = snapshotAIContentEl.querySelector('.aio-text-toggle');
                const textBody = snapshotAIContentEl.querySelector('.aio-text-body');
                if (toggleBtn && textBody) {
                    if (textBody.scrollHeight <= 200) {
                        toggleBtn.style.display = 'none';
                        textBody.classList.add('expanded');
                    }
                    toggleBtn.addEventListener('click', () => {
                        const isExpanded = textBody.classList.toggle('expanded');
                        toggleBtn.textContent = isExpanded ? 'Show less ▲' : 'Show more ▼';
                    });
                }
            }
        }
    }
    const aimPlacesSection = document.getElementById('aimPlacesSection');
    const aimKGSection = document.getElementById('aimKGSection');
    if (aimPlacesSection) aimPlacesSection.style.display = 'none';
    if (aimKGSection) aimKGSection.style.display = 'none';
    if (capture.captureType === 'aim') {
        if (capture.placeEntities && capture.placeEntities.length > 0 && aimPlacesSection) {
            aimPlacesSection.style.display = 'block';
            let placesHtml = `<table class="magi-sources-table"><thead><tr><th>#</th><th>Name</th><th>MID</th><th>Type</th></tr></thead><tbody>`;
            capture.placeEntities.forEach((place, i) => {
                placesHtml += `<tr>
                    <td style="text-align:center;color:var(--text-secondary);font-size:11px;">${i + 1}</td>
                    <td><a href="${escapeHtml(place.href || '')}" target="_blank" style="color:var(--primary);text-decoration:none;">${escapeHtml(place.name || '?')}</a></td>
                    <td style="font-size:11px;color:var(--text-secondary);">${escapeHtml(place.mid || '')}</td>
                    <td style="font-size:11px;"><span style="padding:2px 8px;border-radius:4px;font-size:10px;font-weight:600;background:#e3f0fa;color:#1565c0;">${escapeHtml(place.type || 'place')}</span></td>
                </tr>`;
            });
            placesHtml += '</tbody></table>';
            document.getElementById('aimPlacesContent').innerHTML = placesHtml;
        }
    }
    const productsSection = document.getElementById('productsSection');
    if (productsSection) {
        productsSection.style.display = 'none';
        const productLinks = (capture.entityLinks || []).filter(link => isProductLink(link.href || ''));
        if (productLinks.length > 0) {
            const sidebarCount = productLinks.filter(l => l.inSidebar).length;
            productsSection.style.display = 'block';
            const typeLabel = capture.captureType === 'aim' ? 'AIM' : 'AIO';
            document.getElementById('productsSectionTitle').textContent = `${typeLabel} Products (${productLinks.length})` + (sidebarCount > 0 ? ` · ${sidebarCount} in sidebar` : '');
            let prodHtml = `<table class="magi-sources-table"><thead><tr>
                <th style="width:30px;">#</th><th>Product</th><th>URL</th><th style="width:80px;">Sidebar</th>
            </tr></thead><tbody>`;
            productLinks.forEach((link, i) => {
                const href = cleanGoogleSearchUrl(link.href || '');
                const inSidebar = !!link.inSidebar;
                prodHtml += `<tr>
                    <td style="text-align:center;color:var(--text-secondary);font-size:11px;">${i + 1}</td>
                    <td style="font-weight:500;font-size:12px;">${escapeHtml(link.text || '?')}</td>
                    <td style="font-size:11px;word-break:break-all;"><a href="${escapeHtml(href)}" target="_blank" style="color:var(--primary);text-decoration:none;">${escapeHtml(href)}</a></td>
                    <td style="text-align:center;${inSidebar ? Y_STYLE : N_STYLE}">${inSidebar ? 'y' : 'n'}</td>
                </tr>`;
            });
            prodHtml += '</tbody></table>';
            document.getElementById('productsContent').innerHTML = prodHtml;
        }
    }
    await renderVennSection(query.query, capture.captureType);
}
function formatAIOGeneratedText(rawText) {
    if (!rawText) return '';
    const allLines = rawText.split(/\n/).filter(b => b.trim());
    const seen = new Set();
    const uniqueLines = [];
    for (const line of allLines) {
        const normalized = line.trim().substring(0, 100);
        if (!seen.has(normalized)) {
            seen.add(normalized);
            uniqueLines.push(line.trim());
        }
    }
    const text = uniqueLines.join('\n');
    const hasStructure = /\n## /.test('\n' + text) || /\n- /.test('\n' + text);
    if (hasStructure) {
        return formatStructuredResponse(text);
    }
    return formatLegacyResponse(text);
}
function formatStructuredResponse(text) {
    const lines = text.split('\n');
    let html = '';
    let inList = false;
    for (const line of lines) {
        const trimmed = line.trim();
        if (!trimmed) continue;
        if (trimmed.startsWith('## ')) {
            if (inList) { html += '</ul>'; inList = false; }
            const heading = trimmed.substring(3);
            html += `<h4 style="margin:12px 0 4px 0;font-size:13px;font-weight:700;">${escapeHtml(heading)}</h4>`;
        } else if (trimmed.startsWith('- ')) {
            if (!inList) { html += '<ul style="margin:4px 0;padding-left:20px;">'; inList = true; }
            const item = trimmed.substring(2);
            html += `<li style="margin-bottom:4px;font-size:12px;">${escapeHtml(item)}</li>`;
        } else {
            if (inList) { html += '</ul>'; inList = false; }
            html += `<p style="margin:6px 0;">${escapeHtml(trimmed)}</p>`;
        }
    }
    if (inList) html += '</ul>';
    return html || `<p>${escapeHtml(text)}</p>`;
}
function formatLegacyResponse(text) {
    const sentences = text
        .replace(/([.!?])\s+([A-Z])/g, '$1\n$2')
        .replace(/(\.{3}|…)\s*([A-Z])/g, '$1\n$2')
        .split('\n')
        .map(s => s.trim())
        .filter(s => s.length > 0);
    const sectionHeadingRegex = /^(.{5,80}:)\s*(.*)/;
    let html = '';
    let introSentences = [];
    let sections = [];
    let currentSection = null;
    for (const sentence of sentences) {
        const match = sentence.match(sectionHeadingRegex);
        if (match && !match[1].includes('://') && !/\d:\d/.test(match[1]) && /[A-Z]/.test(match[1][0])) {
            const heading = match[1];
            const rest = match[2] || '';
            const wordCount = heading.replace(/:$/, '').trim().split(/\s+/).length;
            if (wordCount >= 2) {
                if (currentSection) sections.push(currentSection);
                currentSection = { heading, content: rest ? [rest] : [] };
                continue;
            }
        }
        if (currentSection) {
            currentSection.content.push(sentence);
        } else {
            introSentences.push(sentence);
        }
    }
    if (currentSection) sections.push(currentSection);
    if (introSentences.length > 0) {
        const paras = [];
        let cur = [];
        for (const s of introSentences) {
            cur.push(s);
            if (cur.length >= 3 || s.length > 200) { paras.push(cur.join(' ')); cur = []; }
        }
        if (cur.length > 0) paras.push(cur.join(' '));
        html += paras.map(p => `<p>${escapeHtml(p)}</p>`).join('');
    }
    if (sections.length > 0) {
        html += '<ul style="margin:8px 0;padding-left:20px;">';
        for (const sec of sections) {
            const contentText = sec.content.join(' ').trim();
            html += `<li style="margin-bottom:6px;"><strong>${escapeHtml(sec.heading)}</strong>`;
            if (contentText) html += `<br>${escapeHtml(contentText)}`;
            html += '</li>';
        }
        html += '</ul>';
    }
    if (!html) html = `<p>${escapeHtml(text)}</p>`;
    return html;
}
function computeVennData(aioData, domainLevel) {
    const magiUrlMap = new Map();
    const aioUrlMap = new Map();
    if (aioData && aioData.magiInnerHTML) {
        aioData.magiInnerHTML.forEach(magi => {
            const norm = normalizeUrlForComparison(magi.url, domainLevel);
            if (!norm) return;
            if (isExcludedUrl(magi.url)) return;
            if (!magiUrlMap.has(norm)) {
                magiUrlMap.set(norm, { url: magi.url, domain: magi.domain, title: magi.title || '', sourceLabel: magi.sourceLabel || '', count: magi.count });
            }
        });
    }
    const hasSv6KpeCitations = aioData && aioData.citations && aioData.citations.length > 0;
    const aioCitSource = hasSv6KpeCitations
        ? aioData.citations
        : (aioData && aioData.sidebarCards && aioData.sidebarCards.length > 0 ? aioData.sidebarCards : []);
    aioCitSource.forEach(cit => {
        const citUrl = cit.url || '';
        if (!citUrl) return;
        const norm = normalizeUrlForComparison(citUrl, domainLevel);
        if (!norm) return;
        if (!aioUrlMap.has(norm)) {
            let textFragment = cit.textFragment || null;
            if (!textFragment && citUrl.includes('#:~:text=')) {
                const fragMatch = citUrl.match(/#:~:text=(.+)/);
                if (fragMatch) { try { textFragment = decodeURIComponent(fragMatch[1]); } catch (e) { textFragment = fragMatch[1]; } }
            }
            aioUrlMap.set(norm, {
                url: citUrl,
                sourceDomain: cit.sourceDomain || cit.domain || '',
                title: cit.title || '',
                snippet: cit.snippet || '',
                position: cit.position || 0,
                opaqueScore: cit.opaqueScore || null,
                srcId: cit.srcId || null,
                textFragment: textFragment,
                publishedDate: cit.publishedDate || null
            });
        }
    });
    const magiOnly = [];
    const both = [];
    const aioOnly = [];
    for (const [norm, magiInfo] of magiUrlMap) {
        if (aioUrlMap.has(norm)) {
            both.push({ normalized: norm, magi: magiInfo, aio: aioUrlMap.get(norm) });
        } else {
            magiOnly.push({ normalized: norm, magi: magiInfo });
        }
    }
    for (const [norm, citInfo] of aioUrlMap) {
        if (!magiUrlMap.has(norm)) {
            aioOnly.push({ normalized: norm, aio: citInfo });
        }
    }
    return { magiOnly, both, aioOnly, magiTotal: magiUrlMap.size, aioTotal: aioUrlMap.size, hasSv6KpeCitations };
}
async function renderVennSection(queryText, captureType) {
    const section = document.getElementById('vennSection');
    const content = document.getElementById('vennContent');
    const toggleBtn = document.getElementById('vennToggleLevel');
    if (!section || !content) return;
    const type = captureType || 'aio';
    const prefix = type === 'aim' ? 'aim_' : 'aio_';
    let aioData = null;
    try {
        const data = await new Promise(resolve => chrome.storage.local.get(null, resolve));
        for (const [key, value] of Object.entries(data)) {
            if (key.startsWith(prefix) && typeof value === 'object' && value.searchQuery === queryText) {
                aioData = value;
                break;
            }
        }
    } catch (e) {}
    const sectionTitleEl = section.querySelector('.section-title');
    if (sectionTitleEl) {
        sectionTitleEl.innerHTML = type === 'aim'
            ? '<span class="section-icon">&#x1f50d;</span> AIM Sources'
            : '<span class="section-icon">&#x1f50d;</span> Grounding vs Displayed in AIO sidebar card';
    }
    const magiInner = (aioData && aioData.magiInnerHTML) ? aioData.magiInnerHTML.length : 0;
    const aioCitations = (aioData && aioData.citations) ? aioData.citations.length : 0;
    const sidebarItems = type === 'aim'
        ? ((aioData && aioData.sidebarSources) ? aioData.sidebarSources.length : 0)
        : ((aioData && aioData.sidebarCards) ? aioData.sidebarCards.length : 0);
    if (magiInner === 0 && aioCitations === 0 && sidebarItems === 0) {
        section.style.display = 'none';
        return;
    }
    section.style.display = 'block';
    let domainLevel = false;
    let activeStage = null;
    if (type === 'aim') {
        if (toggleBtn) toggleBtn.style.display = 'none';
        const sidebarNormUrls = new Set();
        const aimSidebar = aioData && aioData.sidebarSources ? aioData.sidebarSources : [];
        let sidebarSource;
        if (aimSidebar.length > 0) {
            sidebarSource = aimSidebar;
        } else {
            const citCopy = [...(aioData && aioData.citations ? aioData.citations : [])];
            citCopy.sort((a, b) => (a.position || 9999) - (b.position || 9999));
            sidebarSource = citCopy;
        }
        sidebarSource.forEach(card => {
            const norm = normalizeUrlForComparison(card.url || '', false);
            if (norm) sidebarNormUrls.add(norm);
        });
        const magiLabelMap = new Map();
        if (aioData && aioData.magiInnerHTML && Array.isArray(aioData.magiInnerHTML)) {
            aioData.magiInnerHTML.forEach(m => {
                if (!m.url) return;
                const norm = normalizeUrlForComparison(m.url, false);
                if (norm && m.sourceLabel) magiLabelMap.set(norm, m.sourceLabel);
            });
        }
        const seenNorm = new Set();
        const allUrls = [];
        const citations = aioData && aioData.citations ? aioData.citations : [];
        citations.forEach(c => {
            if (!c.url || isExcludedUrl(c.url)) return;
            const norm = normalizeUrlForComparison(c.url, false);
            if (norm && !seenNorm.has(norm)) {
                seenNorm.add(norm);
                allUrls.push({ url: c.url, domain: c.sourceDomain || extractDomain(c.url) || '', norm, sourceLabel: c.sourceLabel || magiLabelMap.get(norm) || '', publishedDate: c.publishedDate || '' });
            }
        });
        sidebarSource.forEach(s => {
            if (!s.url || isExcludedUrl(s.url)) return;
            const norm = normalizeUrlForComparison(s.url, false);
            if (norm && !seenNorm.has(norm)) {
                seenNorm.add(norm);
                allUrls.push({ url: s.url, domain: s.domain || extractDomain(s.url) || '', norm, sourceLabel: s.sourceLabel || magiLabelMap.get(norm) || '', publishedDate: s.publishedDate || '' });
            }
        });
        // Add product links (from entityLinks) that are in the sidebar
        const productLinks = (aioData && aioData.entityLinks || []).filter(link => isProductLink(link.href || '') && link.inSidebar);
        productLinks.forEach(link => {
            const href = cleanGoogleSearchUrl(link.href || '');
            const norm = normalizeUrlForComparison(href, false);
            if (norm && !seenNorm.has(norm)) {
                seenNorm.add(norm);
                sidebarNormUrls.add(norm);
                allUrls.push({ url: href, domain: extractDomain(href) || '', norm, sourceLabel: link.text || '', publishedDate: '' });
            }
        });
        allUrls.sort((a, b) => {
            const aIn = sidebarNormUrls.has(a.norm) ? 0 : 1;
            const bIn = sidebarNormUrls.has(b.norm) ? 0 : 1;
            return aIn - bIn;
        });
        let tableHtml = '';
        if (allUrls.length > 0) {
            const yStyle = Y_STYLE;
            const nStyle = N_STYLE;
            tableHtml += `<table class="magi-sources-table" style="margin-top:16px;">
                <thead><tr>
                    <th style="width:30px;">#</th>
                    <th>URL</th>
                    <th style="width:100px;">Source</th>
                    <th style="width:80px;">Sidebar</th>
                    <th style="width:100px;">Date</th>
                </tr></thead><tbody>`;
            allUrls.forEach((item, i) => {
                const inSidebar = sidebarNormUrls.has(item.norm);
                tableHtml += `<tr>
                    <td style="text-align:center;color:var(--text-secondary);font-size:11px;">${i + 1}</td>
                    <td style="font-size:11px;word-break:break-all;">${faviconImg(item.domain)}<a href="${escapeHtml(item.url)}" target="_blank" style="color:var(--primary);text-decoration:none;">${escapeHtml(item.url)}</a></td>
                    <td style="font-size:10px;color:var(--text-secondary);">${escapeHtml(cleanSourceLabel(item.sourceLabel))}</td>
                    <td style="text-align:center;${inSidebar ? yStyle : nStyle}">${inSidebar ? 'y' : 'n'}</td>
                    <td style="font-size:11px;color:var(--text-secondary);">${escapeHtml(item.publishedDate || '')}</td>
                </tr>`;
            });
            tableHtml += '</tbody></table>';
        }
        content.innerHTML = tableHtml;
        return;
    }
    if (toggleBtn) toggleBtn.style.display = '';
    function render() {
        const venn = computeVennData(aioData, domainLevel);
        const { magiOnly, both, aioOnly, magiTotal, aioTotal, hasSv6KpeCitations } = venn;
        const sidebarNormUrls = new Set();
        const sidebarRankMap = new Map();
        const sidebarSource = aioData && aioData.sidebarCards ? aioData.sidebarCards : [];
        sidebarSource.forEach((card, idx) => {
            const norm = normalizeUrlForComparison(card.url || '', domainLevel);
            if (norm) {
                sidebarNormUrls.add(norm);
                if (!sidebarRankMap.has(norm)) sidebarRankMap.set(norm, idx + 1);
            }
        });
        const organicNormUrls = new Set();
        if (aioData && aioData.organicUrls && Array.isArray(aioData.organicUrls)) {
            aioData.organicUrls.forEach(entry => {
                const url = typeof entry === 'string' ? entry : (entry.url || '');
                if (!url) return;
                const norm = normalizeUrlForComparison(url, domainLevel);
                if (norm) organicNormUrls.add(norm);
            });
        }
        const intextNormUrls = new Set();
        if (aioData && aioData.externalLinks && Array.isArray(aioData.externalLinks)) {
            aioData.externalLinks.forEach(link => {
                const href = link.href || '';
                if (!href) return;
                const norm = normalizeUrlForComparison(href, domainLevel);
                if (norm) intextNormUrls.add(norm);
            });
        }
        const groundingCount = magiTotal;
        const poolFromGrounding = both.length;
        const videosInjected = aioOnly.length;
        const poolCount = aioTotal;
        const notSelected = magiOnly.length;
        let displayedCount = 0;
        [...both, ...aioOnly].forEach(item => {
            if (item.aio && sidebarNormUrls.has(item.normalized)) displayedCount++;
        });
        const rejectedFromPool = poolCount - displayedCount;
        let wasReranked = false;
        if (hasSv6KpeCitations) {
            [...both, ...aioOnly].forEach(item => {
                if (item.aio && sidebarNormUrls.has(item.normalized)) {
                    const poolPos = item.aio.position || 0;
                    const sidebarRank = sidebarRankMap.get(item.normalized) || 0;
                    if (poolPos > 0 && sidebarRank > 0 && poolPos !== sidebarRank) wasReranked = true;
                }
            });
        }
        let pipelineHtml = `<div class="pipeline">`;
        pipelineHtml += `<div class="pipeline-stage${activeStage === 'grounding' ? ' active' : ''}" data-stage="grounding">
            <div class="pipeline-box grounding"><div class="pipeline-count">${groundingCount}</div><div class="pipeline-label">AI Grounding</div><div class="pipeline-sub">URLs Candidates</div></div></div>`;
        pipelineHtml += `<div class="pipeline-arrow"><div class="pipeline-badge badge-teal${activeStage === 'kept' ? ' active' : ''}" data-filter="kept" style="cursor:pointer;">${poolFromGrounding} kept</div><div class="pipeline-arrow-line">→</div><div style="display:flex;flex-direction:column;align-items:center;gap:6px;"><div class="pipeline-badge badge-red${activeStage === 'dropped' ? ' active' : ''}" data-filter="dropped" style="cursor:pointer;">${notSelected} dropped</div>${videosInjected > 0 ? `<div class="pipeline-badge badge-teal${activeStage === 'injected' ? ' active' : ''}" data-filter="injected" style="cursor:pointer;">+${videosInjected} injected</div>` : ''}</div></div>`;
        pipelineHtml += `<div class="pipeline-stage${activeStage === 'pool' ? ' active' : ''}" data-stage="pool">
            <div class="pipeline-box pool"><div class="pipeline-count">${poolCount}</div><div class="pipeline-label">AIO Pool</div><div class="pipeline-sub">URLs Selected${videosInjected > 0 ? ` · +${videosInjected} injected` : ''}</div></div></div>`;
        pipelineHtml += `<div class="pipeline-arrow"><div class="pipeline-badge badge-teal${activeStage === 'displayed_only' ? ' active' : ''}" data-filter="displayed_only" style="cursor:pointer;">${displayedCount} displayed</div><div class="pipeline-arrow-line">→</div><div class="pipeline-badge badge-red${activeStage === 'rejected' ? ' active' : ''}" data-filter="rejected" style="cursor:pointer;">${rejectedFromPool} rejected</div></div>`;
        pipelineHtml += `<div class="pipeline-stage${activeStage === 'displayed' ? ' active' : ''}" data-stage="displayed">
            <div class="pipeline-box displayed"><div class="pipeline-count">${displayedCount}</div><div class="pipeline-label">Displayed</div><div class="pipeline-sub">Sidebar cards${wasReranked ? ' · reranked' : ''}</div></div></div>`;
        pipelineHtml += `</div>`;
        let detailHtml = '';
        const allItems = [...magiOnly.map(i => ({ ...i, set: 'magi' })), ...both.map(i => ({ ...i, set: 'both' })), ...aioOnly.map(i => ({ ...i, set: 'aio' }))];
        let itemsToShow = allItems;
        if (activeStage === 'grounding') itemsToShow = allItems.filter(i => i.magi);
        else if (activeStage === 'pool') itemsToShow = allItems.filter(i => i.aio);
        else if (activeStage === 'displayed') itemsToShow = allItems.filter(i => i.aio && sidebarNormUrls.has(i.normalized));
        else if (activeStage === 'kept') itemsToShow = allItems.filter(i => i.magi && i.aio);
        else if (activeStage === 'dropped') itemsToShow = allItems.filter(i => i.magi && !i.aio);
        else if (activeStage === 'displayed_only') itemsToShow = allItems.filter(i => i.aio && sidebarNormUrls.has(i.normalized));
        else if (activeStage === 'rejected') itemsToShow = allItems.filter(i => i.aio && !sidebarNormUrls.has(i.normalized));
        else if (activeStage === 'injected') itemsToShow = allItems.filter(i => i.aio && !i.magi);
        if (itemsToShow.length > 0) {
            itemsToShow.sort((a, b) => {
                const aDisplayed = a.aio && sidebarNormUrls.has(a.normalized);
                const bDisplayed = b.aio && sidebarNormUrls.has(b.normalized);
                const aRank = aDisplayed ? (sidebarRankMap.get(a.normalized) || 9999) : 9999;
                const bRank = bDisplayed ? (sidebarRankMap.get(b.normalized) || 9999) : 9999;
                return aRank - bRank;
            });
            detailHtml += `<table class="magi-sources-table" style="margin-top:16px;">
                <thead><tr>
                    <th style="width:30px;">#</th>
                    <th>URL</th>
                    <th style="width:100px;">Source</th>
                    <th style="width:80px;">Grounding</th>
                    <th style="width:80px;">Pool Rank</th>
                    <th style="width:80px;">Displayed Rank</th>
                    <th style="width:80px;">In Organic?</th>
                    <th style="width:80px;">In-text?</th>
                </tr></thead><tbody>`;
            itemsToShow.forEach((item, i) => {
                const url = item.magi ? item.magi.url : (item.aio ? item.aio.url : '');
                const domain = item.magi ? item.magi.domain : (item.aio ? (item.aio.sourceDomain || extractDomain(item.aio.url || '')) : '');
                const inMagi = !!item.magi;
                const inPool = !!item.aio;
                const poolPos = inPool ? (item.aio.position || 0) : 0;
                const inDisplayed = item.aio && sidebarNormUrls.has(item.normalized);
                const displayedRank = inDisplayed ? (sidebarRankMap.get(item.normalized) || 0) : 0;
                const inOrganic = organicNormUrls.has(item.normalized);
                const inIntext = intextNormUrls.has(item.normalized);
                const yStyle = Y_STYLE;
                const nStyle = N_STYLE;
                const srcLabel = cleanSourceLabel(item.magi ? (item.magi.sourceLabel || '') : '');
                detailHtml += `<tr>
                    <td style="text-align:center;color:var(--text-secondary);font-size:11px;">${i + 1}</td>
                    <td style="font-size:11px;word-break:break-all;">${faviconImg(domain)}<a href="${escapeHtml(url)}" target="_blank" style="color:var(--primary);text-decoration:none;">${escapeHtml(domainLevel ? domain : url)}</a></td>
                    <td style="font-size:10px;color:var(--text-secondary);">${escapeHtml(srcLabel)}</td>
                    <td style="text-align:center;${inMagi ? yStyle : nStyle}">${inMagi ? 'y' : 'n'}</td>
                    <td style="text-align:center;${inPool ? yStyle : nStyle}">${poolPos > 0 ? poolPos : (inPool ? '✓' : 'n')}</td>
                    <td style="text-align:center;${inDisplayed ? yStyle : nStyle}">${displayedRank > 0 ? displayedRank : (inDisplayed ? '✓' : 'n')}</td>
                    <td style="text-align:center;${inOrganic ? yStyle : nStyle}">${inOrganic ? 'y' : 'n'}</td>
                    <td style="text-align:center;${inIntext ? yStyle : nStyle}">${inIntext ? 'y' : 'n'}</td>
                </tr>`;
            });
            detailHtml += '</tbody></table>';
        }
        content.innerHTML = pipelineHtml + detailHtml;
        content.querySelectorAll('.pipeline-stage').forEach(stage => {
            stage.addEventListener('click', () => {
                const clickedStage = stage.dataset.stage;
                activeStage = activeStage === clickedStage ? null : clickedStage;
                render();
            });
        });
        content.querySelectorAll('.pipeline-badge[data-filter]').forEach(badge => {
            badge.addEventListener('click', () => {
                const clickedFilter = badge.dataset.filter;
                activeStage = activeStage === clickedFilter ? null : clickedFilter;
                render();
            });
        });
    }
    render();
    if (toggleBtn) {
        toggleBtn.textContent = 'Domain level';
        const newBtn = toggleBtn.cloneNode(true);
        toggleBtn.parentNode.replaceChild(newBtn, toggleBtn);
        newBtn.addEventListener('click', () => {
            domainLevel = !domainLevel;
            newBtn.textContent = domainLevel ? 'URL level' : 'Domain level';
            render();
        });
    }
}
async function computeGlobalDomainData(overrideType) {
    const allData = await new Promise(resolve => chrome.storage.local.get(null, resolve));
    const magiDomains = new Map();
    const magiUrls = new Map();
    const citationDomains = new Map();
    const citationUrls = new Map();
    const intextDomains = new Map();
    const intextUrls = new Map();
    const effectiveType = overrideType || currentDataType;
    const activePrefix = effectiveType === 'aim' ? 'aim_' : 'aio_';
    for (const [key, value] of Object.entries(allData)) {
        if (!key.startsWith(activePrefix) || typeof value !== 'object') continue;
        if (value.magiInnerHTML && Array.isArray(value.magiInnerHTML)) {
            value.magiInnerHTML.forEach(magi => {
                const realUrl = magi.url || '';
                if (!realUrl || realUrl.startsWith('/')) return;
                if (isExcludedUrl(realUrl)) return;
                const domain = magi.domain || extractDomain(realUrl);
                const normUrl = normalizeUrlForComparison(realUrl, false);
                if (!domain) return;
                magiDomains.set(domain, (magiDomains.get(domain) || 0) + 1);
                if (!magiUrls.has(normUrl)) {
                    magiUrls.set(normUrl, { url: realUrl, domain, count: 1, sourceLabel: magi.sourceLabel || '' });
                } else {
                    magiUrls.get(normUrl).count++;
                }
            });
        }
        const citSource = effectiveType === 'aim'
            ? [...(value.citations || []), ...(value.sidebarSources || [])]
            : ((value.sidebarCards && value.sidebarCards.length > 0) ? value.sidebarCards : (value.citations || []));
        if (Array.isArray(citSource)) {
            citSource.forEach(cit => {
                const citUrl = cit.url || '';
                if (!citUrl || isExcludedUrl(citUrl)) return;
                const domain = cit.sourceDomain || cit.domain || extractDomain(citUrl) || '';
                const normUrl = normalizeUrlForComparison(citUrl, false);
                if (!domain) return;
                citationDomains.set(domain, (citationDomains.get(domain) || 0) + 1);
                if (!citationUrls.has(normUrl)) {
                    citationUrls.set(normUrl, { url: citUrl, domain, count: 1, sourceLabel: cit.sourceLabel || '' });
                } else {
                    const existing = citationUrls.get(normUrl);
                    existing.count++;
                    if (!existing.sourceLabel && cit.sourceLabel) existing.sourceLabel = cit.sourceLabel;
                }
            });
        }
        if (value.externalLinks && Array.isArray(value.externalLinks)) {
            value.externalLinks.forEach(link => {
                if (!link.href || isExcludedUrl(link.href)) return;
                const domain = link.domain || '';
                if (!domain) return;
                const normUrl = normalizeUrlForComparison(link.href, false);
                intextDomains.set(domain, (intextDomains.get(domain) || 0) + 1);
                if (!intextUrls.has(normUrl)) {
                    intextUrls.set(normUrl, { url: link.href, domain, count: 1 });
                } else {
                    intextUrls.get(normUrl).count++;
                }
            });
        }
    }
    return { magiDomains, citationDomains, intextDomains, magiUrls, citationUrls, intextUrls };
}
function classifyAndMerge(magiMap, citationMap, intextMap) {
    const merged = new Map();
    function addToMerged(key, val, flag, countField) {
        const count = typeof val === 'number' ? val : val.count;
        if (merged.has(key)) {
            const existing = merged.get(key);
            existing[countField] += count;
            existing[flag] = true;
        } else {
            merged.set(key, {
                key,
                domain: typeof val === 'object' ? val.domain : key,
                url: typeof val === 'object' ? val.url : null,
                sourceLabel: typeof val === 'object' ? (val.sourceLabel || '') : '',
                inMagi: false, inCitations: false, inIntext: false,
                [flag]: true,
                magiCount: 0, citationCount: 0, intextCount: 0,
                [countField]: count
            });
        }
    }
    for (const [key, val] of magiMap) addToMerged(key, val, 'inMagi', 'magiCount');
    for (const [key, val] of citationMap) addToMerged(key, val, 'inCitations', 'citationCount');
    for (const [key, val] of intextMap) addToMerged(key, val, 'inIntext', 'intextCount');
    return Array.from(merged.values()).map(item => {
        if (item.inMagi && (item.inCitations || item.inIntext)) item.source = 'both';
        else if (item.inMagi) item.source = 'magi';
        else if (item.inIntext && !item.inCitations) item.source = 'intext';
        else item.source = 'aio';
        return item;
    });
}
function getFilterCount(item, filter) {
    if (filter === 'magi') return item.magiCount;
    if (filter === 'citations') return item.citationCount;
    if (filter === 'intext') return item.intextCount;
    if (filter === 'aio') return item.citationCount + item.intextCount;
    if (filter === 'both') return item.magiCount + item.citationCount + item.intextCount;
    return item.magiCount + item.citationCount + item.intextCount;
}
function filterItems(items, filter) {
    if (filter === 'magi') return items.filter(i => i.inMagi);
    if (filter === 'aio') return items.filter(i => i.inCitations || i.inIntext);
    if (filter === 'both') return items.filter(i => i.inMagi && (i.inCitations || i.inIntext));
    if (filter === 'citations') return items.filter(i => i.inCitations);
    if (filter === 'intext') return items.filter(i => i.inIntext);
    return [...items];
}
function renderBarChart(container, items, filter, maxBars, showDomain, chartId) {
    let filtered = filterItems(items, filter);
    filtered.sort((a, b) => getFilterCount(b, filter) - getFilterCount(a, filter) || a.key.localeCompare(b.key));
    const top = filtered.slice(0, maxBars);
    if (top.length === 0) {
        container.innerHTML = '<div class="no-data" style="padding:20px;">No data for this filter</div>';
        return;
    }
    const maxCount = getFilterCount(top[0], filter);
    const maxBarHeight = 160;
    let html = `<div style="font-size:12px;font-weight:600;color:var(--text-secondary);margin-bottom:8px;">Top ${Math.min(maxBars, filtered.length)} ${showDomain ? 'Domains' : 'URLs'} by Count</div>`;
    html += '<div class="bar-chart">';
    top.forEach(item => {
        const count = getFilterCount(item, filter);
        const barH = maxCount > 0 ? Math.max(8, Math.round((count / maxCount) * maxBarHeight)) : 8;
        let colorClass = 'bar-aio';
        if (item.source === 'both') colorClass = 'bar-both';
        else if (item.source === 'magi') colorClass = 'bar-magi';
        else if (item.source === 'intext') colorClass = 'bar-intext';
        const displayLabel = showDomain ? item.key : (item.domain || item.key);
        const faviconDomain = showDomain ? item.key : (item.domain || '');
        const tooltip = showDomain ? item.key : (item.url || item.key);
        html += `<div class="bar-col" data-bar-key="${escapeHtml(item.key)}" data-chart-id="${chartId}" style="cursor:pointer;" title="${escapeHtml(tooltip)}">
            <div class="bar-count">${count}</div>
            <div class="bar-rect ${colorClass}" style="height:${barH}px;"></div>
            <img class="bar-favicon" src="https://www.google.com/s2/favicons?domain=${encodeURIComponent(faviconDomain)}&sz=16" alt="" onerror="this.style.visibility='hidden'">
            <div class="bar-label">${escapeHtml(displayLabel)}</div>
        </div>`;
    });
    html += '</div>';
    container.innerHTML = html;
    if (showDomain) {
        container.querySelectorAll('.bar-col[data-bar-key]').forEach(col => {
            col.addEventListener('click', () => {
                const domain = col.dataset.barKey;
                showDomainUrlsModal(domain, col.dataset.chartId);
            });
        });
    }
}
function showChartModal(title, tableHtml, copyText) {
    const overlay = document.getElementById('chartModalOverlay');
    document.getElementById('chartModalTitle').textContent = title;
    document.getElementById('chartModalBody').innerHTML = tableHtml;
    overlay.classList.add('visible');
    const copyBtn = document.getElementById('chartModalCopy');
    copyBtn.textContent = 'Copy';
    copyBtn.classList.remove('copied');
    copyBtn.onclick = () => {
        navigator.clipboard.writeText(copyText).then(() => {
            copyBtn.textContent = 'Copied!';
            copyBtn.classList.add('copied');
            setTimeout(() => { copyBtn.textContent = 'Copy'; copyBtn.classList.remove('copied'); }, 1500);
        });
    };
}
function closeChartModal() {
    document.getElementById('chartModalOverlay').classList.remove('visible');
}
function showDomainUrlsModal(domain, chartId) {
    const urlItems = _chartUrlItems.filter(i => i.domain === domain || i.key.startsWith(domain));
    let tableHtml = '<table><thead><tr><th>#</th><th>URL</th><th style="width:90px;">Source</th><th style="width:60px;">Grounding</th><th style="width:60px;">Sidebar</th><th style="width:60px;">In-text</th></tr></thead><tbody>';
    let copyLines = ['#\tURL\tSource\tGrounding\tSidebar\tIn-text'];
    urlItems.sort((a, b) => (b.magiCount + b.citationCount + b.intextCount) - (a.magiCount + a.citationCount + a.intextCount));
    urlItems.forEach((item, i) => {
        const url = item.url || item.key;
        const sl = cleanSourceLabel(item.sourceLabel);
        const g = item.inMagi ? 'y' : 'n';
        const s = item.inCitations ? 'y' : 'n';
        const t = item.inIntext ? 'y' : 'n';
        const yStyle = Y_STYLE;
        const nStyle = N_STYLE;
        tableHtml += `<tr>
            <td style="text-align:center;color:var(--text-secondary);">${i + 1}</td>
            <td><a href="${escapeHtml(url)}" target="_blank" style="color:var(--primary);text-decoration:none;">${escapeHtml(url)}</a></td>
            <td style="font-size:10px;color:var(--text-secondary);">${escapeHtml(sl)}</td>
            <td style="text-align:center;${g === 'y' ? yStyle : nStyle}">${g}</td>
            <td style="text-align:center;${s === 'y' ? yStyle : nStyle}">${s}</td>
            <td style="text-align:center;${t === 'y' ? yStyle : nStyle}">${t}</td>
        </tr>`;
        copyLines.push(`${i + 1}\t${url}\t${sl}\t${g}\t${s}\t${t}`);
    });
    tableHtml += '</tbody></table>';
    showChartModal(`URLs for ${domain} (${urlItems.length})`, tableHtml, copyLines.join('\n'));
}
function showCopyAllUrlsModal(items, title) {
    let sorted = [...items].sort((a, b) => (a.domain || a.key).localeCompare(b.domain || b.key) || (a.url || a.key).localeCompare(b.url || b.key));
    let tableHtml = '<table><thead><tr><th>#</th><th>URL</th><th>Domain</th><th style="width:90px;">Source</th><th style="width:60px;">Grounding</th><th style="width:60px;">Sidebar</th><th style="width:60px;">In-text</th></tr></thead><tbody>';
    let copyLines = ['#\tURL\tDomain\tSource\tGrounding\tSidebar\tIn-text'];
    sorted.forEach((item, i) => {
        const url = item.url || item.key;
        const domain = item.domain || item.key;
        const sl = cleanSourceLabel(item.sourceLabel);
        const g = item.inMagi ? 'y' : 'n';
        const s = item.inCitations ? 'y' : 'n';
        const t = item.inIntext ? 'y' : 'n';
        const yStyle = Y_STYLE;
        const nStyle = N_STYLE;
        tableHtml += `<tr>
            <td style="text-align:center;color:var(--text-secondary);">${i + 1}</td>
            <td><a href="${escapeHtml(url)}" target="_blank" style="color:var(--primary);text-decoration:none;font-size:11px;">${escapeHtml(url)}</a></td>
            <td style="font-size:11px;">${escapeHtml(domain)}</td>
            <td style="font-size:10px;color:var(--text-secondary);">${escapeHtml(sl)}</td>
            <td style="text-align:center;${g === 'y' ? yStyle : nStyle}">${g}</td>
            <td style="text-align:center;${s === 'y' ? yStyle : nStyle}">${s}</td>
            <td style="text-align:center;${t === 'y' ? yStyle : nStyle}">${t}</td>
        </tr>`;
        copyLines.push(`${i + 1}\t${url}\t${domain}\t${sl}\t${g}\t${s}\t${t}`);
    });
    tableHtml += '</tbody></table>';
    showChartModal(title + ` (${sorted.length})`, tableHtml, copyLines.join('\n'));
}
let _chartDomainItems = [];
let _chartUrlItems = [];
let _chartListenersAttached = false;
async function renderGlobalCharts() {
    const data = await computeGlobalDomainData('aio');
    _chartDomainItems = classifyAndMerge(data.magiDomains, data.citationDomains, data.intextDomains);
    _chartUrlItems = classifyAndMerge(data.magiUrls, data.citationUrls, data.intextUrls);
    renderBarChart(document.getElementById('domainChartContent'), _chartDomainItems, 'all', 20, true, 'domains');
    renderBarChart(document.getElementById('urlChartContent'), _chartUrlItems, 'all', 20, false, 'urls');
    document.querySelectorAll('#domainChartFilters .chart-filter-btn').forEach(b => b.classList.toggle('active', b.dataset.filter === 'all'));
    document.querySelectorAll('#urlChartFilters .chart-filter-btn').forEach(b => b.classList.toggle('active', b.dataset.filter === 'all'));
    if (_chartListenersAttached) return;
    _chartListenersAttached = true;
    document.getElementById('domainChartFilters').addEventListener('click', (e) => {
        const btn = e.target.closest('.chart-filter-btn');
        if (!btn) return;
        document.querySelectorAll('#domainChartFilters .chart-filter-btn').forEach(b => b.classList.remove('active'));
        btn.classList.add('active');
        renderBarChart(document.getElementById('domainChartContent'), _chartDomainItems, btn.dataset.filter, 20, true, 'domains');
    });
    document.getElementById('urlChartFilters').addEventListener('click', (e) => {
        const btn = e.target.closest('.chart-filter-btn');
        if (!btn) return;
        document.querySelectorAll('#urlChartFilters .chart-filter-btn').forEach(b => b.classList.remove('active'));
        btn.classList.add('active');
        renderBarChart(document.getElementById('urlChartContent'), _chartUrlItems, btn.dataset.filter, 20, false, 'urls');
    });
    document.getElementById('domainCopyAllBtn').addEventListener('click', () => {
        showCopyAllUrlsModal(_chartUrlItems, 'All URLs — Domains');
    });
    document.getElementById('urlCopyAllBtn').addEventListener('click', () => {
        showCopyAllUrlsModal(_chartUrlItems, 'All URLs');
    });
    document.getElementById('chartModalClose').addEventListener('click', closeChartModal);
    document.getElementById('chartModalOverlay').addEventListener('click', (e) => {
        if (e.target === e.currentTarget) closeChartModal();
    });
}
async function renderAllAIOEntities() {
    const container = document.getElementById('allAIOEntitiesContent');
    if (!container) return;
    const allData = await new Promise(resolve => chrome.storage.local.get(null, resolve));
    const entityMap = new Map();
    for (const [key, value] of Object.entries(allData)) {
        if (!key.startsWith('aio_') || typeof value !== 'object') continue;
        const q = value.searchQuery || '?';
        if (value.entityLinks && Array.isArray(value.entityLinks)) {
            value.entityLinks.forEach(link => {
                const href = cleanGoogleSearchUrl(link.href || '');
                if (!href) return;
                if (isProductLink(href)) return;
                if (!entityMap.has(href)) {
                    entityMap.set(href, { text: link.text || '', href, queries: [q] });
                } else {
                    const existing = entityMap.get(href);
                    if (!existing.queries.includes(q)) existing.queries.push(q);
                }
            });
        }
    }
    if (entityMap.size === 0) {
        container.innerHTML = '<div class="no-data">No AIO entities found</div>';
        return;
    }
    const entities = Array.from(entityMap.values()).sort((a, b) => b.queries.length - a.queries.length || a.text.localeCompare(b.text));
    let html = `<table class="magi-sources-table">
        <thead><tr>
            <th style="width:30px;">#</th>
            <th>Entity</th>
            <th style="width:80px;">Occurrences</th>
            <th>Queries</th>
        </tr></thead><tbody>`;
    entities.forEach((ent, i) => {
        html += `<tr>
            <td style="text-align:center;color:var(--text-secondary);font-size:11px;">${i + 1}</td>
            <td><a href="${escapeHtml(ent.href)}" target="_blank" style="color:var(--primary);text-decoration:none;font-weight:500;font-size:12px;" title="${escapeHtml(ent.href)}">${escapeHtml(ent.text)}</a></td>
            <td style="text-align:center;font-weight:600;font-size:12px;">${ent.queries.length}</td>
            <td style="font-size:11px;">${renderQueryLinks(ent.queries)}</td>
        </tr>`;
    });
    html += '</tbody></table>';
    container.innerHTML = html;
    wireQueryNavLinks(container);
    const copyBtn = document.getElementById('entitiesCopyBtn');
    if (copyBtn) {
        copyBtn.onclick = () => {
            const lines = ['#\tEntity\tURL\tOccurrences\tQueries'];
            entities.forEach((ent, i) => {
                lines.push(`${i + 1}\t${ent.text}\t${ent.href}\t${ent.queries.length}\t${ent.queries.join(', ')}`);
            });
            navigator.clipboard.writeText(lines.join('\n')).then(() => {
                copyBtn.textContent = 'Copied!';
                setTimeout(() => { copyBtn.textContent = 'Copy'; }, 1500);
            });
        };
    }
}
let _aiDomainsItems = [];
let _aiDomainsListenerAttached = false;
async function renderAIDomains() {
    const container = document.getElementById('aiDomainsContent');
    if (!container) return;
    const allData = await new Promise(resolve => chrome.storage.local.get(null, resolve));
    const citationUrlSet = new Map();
    const magiUrlSet = new Map();
    const intextUrlSet = new Map();
    for (const [key, value] of Object.entries(allData)) {
        if (!key.startsWith('aio_') || typeof value !== 'object') continue;
        const q = value.searchQuery || '?';
        const citSource = (value.sidebarCards && value.sidebarCards.length > 0) ? value.sidebarCards : (value.citations || []);
        if (Array.isArray(citSource)) {
            citSource.forEach(cit => {
                const citUrl = cit.url || '';
                if (!citUrl || isExcludedUrl(citUrl)) return;
                const normUrl = normalizeUrlForComparison(citUrl, false);
                if (!citationUrlSet.has(normUrl)) {
                    citationUrlSet.set(normUrl, {
                        url: citUrl,
                        sourceDomain: cit.sourceDomain || cit.domain || '',
                        title: cit.title || '',
                        snippet: cit.snippet || '',
                        position: cit.position || 0,
                        opaqueScore: cit.opaqueScore || null,
                        searchQuery: q
                    });
                }
            });
        }
        if (value.citations && Array.isArray(value.citations) && citSource !== value.citations) {
            value.citations.forEach(cit => {
                const citUrl = cit.url || '';
                if (!citUrl || isExcludedUrl(citUrl) || !cit.opaqueScore) return;
                const normUrl = normalizeUrlForComparison(citUrl, false);
                const existing = citationUrlSet.get(normUrl);
                if (existing && !existing.opaqueScore) {
                    existing.opaqueScore = cit.opaqueScore;
                }
            });
        }
        if (value.externalLinks && Array.isArray(value.externalLinks)) {
            value.externalLinks.forEach(link => {
                if (!link.href || isExcludedUrl(link.href)) return;
                const normUrl = normalizeUrlForComparison(link.href, false);
                if (!intextUrlSet.has(normUrl)) {
                    intextUrlSet.set(normUrl, { url: link.href, domain: link.domain || '', searchQuery: q });
                }
            });
        }
        if (value.magiInnerHTML && Array.isArray(value.magiInnerHTML)) {
            value.magiInnerHTML.forEach(magi => {
                const realUrl = magi.url || '';
                if (!realUrl || realUrl.startsWith('/')) return;
                if (isExcludedUrl(realUrl)) return;
                const normUrl = normalizeUrlForComparison(realUrl, false);
                if (!magiUrlSet.has(normUrl)) {
                    magiUrlSet.set(normUrl, { realUrl, domain: magi.domain || extractDomain(realUrl), searchQuery: q });
                }
            });
        }
    }
    _aiDomainsItems = [];
    const allNorms = new Set([...citationUrlSet.keys(), ...magiUrlSet.keys(), ...intextUrlSet.keys()]);
    for (const norm of allNorms) {
        const cit = citationUrlSet.get(norm) || null;
        const magi = magiUrlSet.get(norm) || null;
        const intext = intextUrlSet.get(norm) || null;
        const inMagi = !!magi;
        const inCitations = !!cit;
        const inIntext = !!intext;
        let source;
        if (inMagi && (inCitations || inIntext)) source = 'both';
        else if (inMagi) source = 'magi';
        else if (inIntext && !inCitations) source = 'intext';
        else source = 'aio';
        const url = magi ? magi.realUrl : (cit ? cit.url : (intext ? intext.url : norm));
        const domain = magi ? magi.domain : (cit ? (cit.sourceDomain || extractDomain(cit.url || '')) : (intext ? intext.domain : ''));
        const title = cit ? (cit.title || '') : '';
        const position = cit ? (cit.position || '') : '';
        const opaqueScore = cit ? (cit.opaqueScore || null) : null;
        const query = magi ? magi.searchQuery : (cit ? cit.searchQuery : (intext ? intext.searchQuery : '?'));
        _aiDomainsItems.push({ norm, url, domain, title, position, opaqueScore, source, query, inMagi, inCitations, inIntext });
    }
    _aiDomainsItems.sort((a, b) => (a.query || '').localeCompare(b.query || '') || (a.domain || '').localeCompare(b.domain || ''));
    renderAIDomainsTable('all');
    document.querySelectorAll('#aiDomainsFilters .chart-filter-btn').forEach(b => b.classList.toggle('active', b.dataset.filter === 'all'));
    if (_aiDomainsListenerAttached) return;
    _aiDomainsListenerAttached = true;
    document.getElementById('aiDomainsFilters').addEventListener('click', (e) => {
        const btn = e.target.closest('.chart-filter-btn');
        if (!btn) return;
        document.querySelectorAll('#aiDomainsFilters .chart-filter-btn').forEach(b => b.classList.remove('active'));
        btn.classList.add('active');
        renderAIDomainsTable(btn.dataset.filter);
    });
}
function _getUrlFragment(url) {
    try {
        const hash = new URL(url).hash;
        if (!hash) return '';
        const raw = hash.slice(1);
        try { return decodeURIComponent(raw); } catch { return raw; }
    } catch { return ''; }
}
function _aiDomainsOriginLabel(item) {
    return item.source === 'both' ? 'Both' : (item.source === 'magi' ? 'Grounding' : (item.source === 'intext' ? 'In-text' : 'AIO'));
}
const AI_DOMAINS_DISPLAY_LIMIT = 200;
function renderAIDomainsTable(filter) {
    const container = document.getElementById('aiDomainsContent');
    if (!container) return;
    let items = _aiDomainsItems;
    if (filter === 'magi') items = _aiDomainsItems.filter(i => i.inMagi);
    else if (filter === 'aio') items = _aiDomainsItems.filter(i => i.inCitations || i.inIntext);
    else if (filter === 'both') items = _aiDomainsItems.filter(i => i.inMagi && (i.inCitations || i.inIntext));
    else if (filter === 'citations') items = _aiDomainsItems.filter(i => i.inCitations);
    else if (filter === 'intext') items = _aiDomainsItems.filter(i => i.inIntext);
    if (items.length === 0) {
        container.innerHTML = '<div class="no-data">No data for this filter</div>';
        return;
    }
    // Sort by score descending (nulls last)
    items = items.slice().sort((a, b) => {
        const sa = a.opaqueScore || 0;
        const sb = b.opaqueScore || 0;
        return sb - sa;
    });
    const totalCount = items.length;
    const truncated = totalCount > AI_DOMAINS_DISPLAY_LIMIT;
    const displayItems = truncated ? items.slice(0, AI_DOMAINS_DISPLAY_LIMIT) : items;
    const sourceBadge = { magi: 'background:#fff3e0;color:#e65100;', aio: 'background:#e0f2f1;color:#00695c;', both: 'background:#e8eaf6;color:#283593;', intext: 'background:#e3f0fa;color:#1565c0;' };
    let html = `<table class="magi-sources-table">
        <thead><tr>
            <th style="width:30px;">#</th>
            <th>Source</th>
            <th>Title</th>
            <th style="width:70px;cursor:help;" title="Position as found in Google's raw citation data (entry[8]). Differs from the sequential # rank shown in the first column.">Orig. Pos.</th>
            <th style="width:60px;cursor:help;" title="Opaque score from Google's grounding data. Analysis suggests it correlates with passage-level grounding confidence — how well the source content matches the generated response — rather than ranking or domain authority.">Score?</th>
            <th style="width:70px;">Origin</th>
            <th>Fragment</th>
            <th>Query</th>
        </tr></thead><tbody>`;
    displayItems.forEach((item, i) => {
        const srcStyle = sourceBadge[item.source] || '';
        const srcLabel = _aiDomainsOriginLabel(item);
        const fragment = _getUrlFragment(item.url);
        html += `<tr>
            <td style="text-align:center;color:var(--text-secondary);font-size:11px;">${i + 1}</td>
            <td style="font-size:12px;font-weight:500;">${faviconImg(item.domain)}<a href="${escapeHtml(item.url)}" target="_blank" style="color:var(--primary);text-decoration:none;font-size:12px;" title="${escapeHtml(item.url)}">${escapeHtml(item.domain || '—')}</a></td>
            <td style="font-size:12px;">${escapeHtml(item.title)}</td>
            <td style="text-align:center;font-size:11px;color:var(--text-secondary);">${item.position || '—'}</td>
            <td style="text-align:center;font-size:11px;color:var(--text-secondary);">${item.opaqueScore || '—'}</td>
            <td style="text-align:center;"><span style="${srcStyle}padding:2px 8px;border-radius:4px;font-size:10px;font-weight:600;">${srcLabel}</span></td>
            <td style="font-size:11px;color:var(--text-secondary);max-width:150px;overflow:hidden;text-overflow:ellipsis;white-space:nowrap;" title="${escapeHtml(fragment)}">${escapeHtml(fragment) || '—'}</td>
            <td style="font-size:11px;">${renderSingleQueryLink(item.query)}</td>
        </tr>`;
    });
    html += '</tbody></table>';
    if (truncated) {
        html += `<div style="margin-top:10px;padding:10px 14px;background:var(--hover-bg, #f5f5f5);border-radius:6px;font-size:12px;color:var(--text-secondary);display:flex;align-items:center;justify-content:space-between;gap:10px;">
            <span>Showing ${AI_DOMAINS_DISPLAY_LIMIT} of ${totalCount} URLs (sorted by score descending). The list is truncated.</span>
            <button class="copy-all-urls-btn" id="aiDomainsCsvBtnBottom" title="Download all ${totalCount} URLs as CSV">Download full CSV (${totalCount})</button>
        </div>`;
    }
    container.innerHTML = html;
    wireQueryNavLinks(container);
    // Copy button — copies displayed items
    const copyBtn = document.getElementById('aiDomainsCopyBtn');
    if (copyBtn) {
        copyBtn.onclick = () => {
            const lines = ['#\tURL\tDomain\tTitle\tOrig. Pos.\tScore\tOrigin\tFragment\tQuery'];
            displayItems.forEach((item, i) => {
                lines.push(`${i + 1}\t${item.url}\t${item.domain || ''}\t${item.title || ''}\t${item.position || ''}\t${item.opaqueScore || ''}\t${_aiDomainsOriginLabel(item)}\t${_getUrlFragment(item.url)}\t${item.query || ''}`);
            });
            navigator.clipboard.writeText(lines.join('\n')).then(() => {
                copyBtn.textContent = 'Copied!';
                setTimeout(() => { copyBtn.textContent = 'Copy'; }, 1500);
            });
        };
    }
    // CSV download helper — exports ALL items (not truncated)
    const downloadCsv = (feedbackBtn) => {
        const csvEscape = (val) => {
            const s = String(val == null ? '' : val);
            return s.includes(',') || s.includes('"') || s.includes('\n') ? '"' + s.replace(/"/g, '""') + '"' : s;
        };
        const header = ['#','URL','Domain','Title','Orig. Pos.','Score','Origin','Grounding only','AIO only','Both','AIO Sidebar citations','AIO In-text links','Fragment','Query'];
        const rows = [header.join(',')];
        items.forEach((item, i) => {
            const isGroundingOnly = item.inMagi && !item.inCitations && !item.inIntext;
            const isAIOOnly = !item.inMagi && (item.inCitations || item.inIntext);
            const isBoth = item.inMagi && (item.inCitations || item.inIntext);
            rows.push([
                i + 1,
                csvEscape(item.url),
                csvEscape(item.domain || ''),
                csvEscape(item.title || ''),
                csvEscape(item.position || ''),
                csvEscape(item.opaqueScore || ''),
                csvEscape(_aiDomainsOriginLabel(item)),
                isGroundingOnly ? 'Yes' : 'No',
                isAIOOnly ? 'Yes' : 'No',
                isBoth ? 'Yes' : 'No',
                item.inCitations ? 'Yes' : 'No',
                item.inIntext ? 'Yes' : 'No',
                csvEscape(_getUrlFragment(item.url)),
                csvEscape(item.query || '')
            ].join(','));
        });
        const blob = new Blob(['\uFEFF' + rows.join('\n')], { type: 'text/csv;charset=utf-8;' });
        const a = document.createElement('a');
        a.href = URL.createObjectURL(blob);
        a.download = 'ai_domains_' + filter + '_' + new Date().toISOString().slice(0, 10) + '.csv';
        a.click();
        URL.revokeObjectURL(a.href);
        if (feedbackBtn) {
            const orig = feedbackBtn.textContent;
            feedbackBtn.textContent = 'Done!';
            setTimeout(() => { feedbackBtn.textContent = orig; }, 1500);
        }
    };
    // Wire both CSV buttons (top + bottom)
    const csvBtn = document.getElementById('aiDomainsCsvBtn');
    if (csvBtn) csvBtn.onclick = () => downloadCsv(csvBtn);
    const csvBtnBottom = document.getElementById('aiDomainsCsvBtnBottom');
    if (csvBtnBottom) csvBtnBottom.onclick = () => downloadCsv(csvBtnBottom);
}
async function renderFreshness(containerId, prefix) {
    const container = document.getElementById(containerId);
    if (!container) return;
    container.innerHTML = '<div style="color:var(--text-secondary);font-size:12px;">Loading...</div>';
    const allData = await new Promise(resolve => chrome.storage.local.get(null, resolve));
    const now = new Date();
    const oneYearAgo = new Date(now);
    oneYearAgo.setFullYear(oneYearAgo.getFullYear() - 1);
    const allDates = [];
    const seenUrls = new Set();
    for (const [key, value] of Object.entries(allData)) {
        if (!key.startsWith(prefix) || typeof value !== 'object') continue;
        const sources = [];
        if (prefix === 'aio_') {
            sources.push(...(value.citations || []), ...(value.sidebarCards || []));
        } else {
            sources.push(...(value.citations || []), ...(value.sidebarSources || []));
        }
        sources.forEach(src => {
            const url = src.url || '';
            if (!url || isExcludedUrl(url)) return;
            const norm = normalizeUrlForComparison(url, false);
            if (seenUrls.has(norm)) return;
            seenUrls.add(norm);
            const dateStr = src.publishedDate || '';
            if (!dateStr) return;
            const parsed = parseFuzzyDate(dateStr);
            if (parsed) allDates.push(parsed);
        });
    }
    const totalWithDate = allDates.length;
    const totalUrls = seenUrls.size;
    if (totalUrls === 0) {
        container.innerHTML = '<div class="no-data" style="padding:16px;">No URLs found</div>';
        return;
    }
    let under1y = 0, over1y = 0;
    allDates.forEach(d => {
        if (d >= oneYearAgo) under1y++;
        else over1y++;
    });
    const noDate = totalUrls - totalWithDate;
    const monthCounts = new Map();
    const fourYearsAgo = new Date(now);
    fourYearsAgo.setMonth(fourYearsAgo.getMonth() - 47);
    fourYearsAgo.setDate(1);
    const months = [];
    for (let i = 0; i < 48; i++) {
        const d = new Date(fourYearsAgo);
        d.setMonth(d.getMonth() + i);
        const mKey = `${d.getFullYear()}-${String(d.getMonth() + 1).padStart(2, '0')}`;
        months.push(mKey);
        monthCounts.set(mKey, 0);
    }
    allDates.forEach(d => {
        const mKey = `${d.getFullYear()}-${String(d.getMonth() + 1).padStart(2, '0')}`;
        if (monthCounts.has(mKey)) {
            monthCounts.set(mKey, monthCounts.get(mKey) + 1);
        }
    });
    const pctUnder = totalWithDate > 0 ? Math.round((under1y / totalWithDate) * 100) : 0;
    const pctOver = totalWithDate > 0 ? Math.round((over1y / totalWithDate) * 100) : 0;
    const pctDated = totalUrls > 0 ? Math.round((totalWithDate / totalUrls) * 100) : 0;
    const pctNoDate = totalUrls > 0 ? Math.round((noDate / totalUrls) * 100) : 0;
    let html = `<div class="freshness-stats">
        <div class="freshness-stat-box">
            <div class="freshness-stat-value" style="color:var(--text-primary);">${totalUrls}</div>
            <div class="freshness-stat-label">Total URLs</div>
        </div>
        <div class="freshness-stat-box">
            <div class="freshness-stat-value" style="color:var(--primary);">${totalWithDate}<span class="freshness-stat-pct" style="color:var(--primary);">(${pctDated}%)</span></div>
            <div class="freshness-stat-label">Dated URLs</div>
        </div>
        <div class="freshness-stat-box">
            <div class="freshness-stat-value" style="color:var(--success);">${under1y}<span class="freshness-stat-pct" style="color:var(--success);">(${pctUnder}%)</span></div>
            <div class="freshness-stat-label">&lt; 1 year old</div>
        </div>
        <div class="freshness-stat-box">
            <div class="freshness-stat-value" style="color:var(--warning);">${over1y}<span class="freshness-stat-pct" style="color:var(--warning);">(${pctOver}%)</span></div>
            <div class="freshness-stat-label">&gt; 1 year old</div>
        </div>
        <div class="freshness-stat-box">
            <div class="freshness-stat-value" style="color:var(--text-secondary);">${noDate}<span class="freshness-stat-pct" style="color:var(--text-secondary);">(${pctNoDate}%)</span></div>
            <div class="freshness-stat-label">No date</div>
        </div>
    </div>`;
    if (totalWithDate > 0) {
        html += `<div style="font-size:12px;font-weight:600;color:var(--text-secondary);margin-bottom:8px;">Articles per month (last 48 months)</div>`;
        html += renderTrendSVG(months, monthCounts);
    }
    container.innerHTML = html;
}
function parseFuzzyDate(dateStr) {
    if (!dateStr) return null;
    const d = new Date(dateStr);
    if (!isNaN(d.getTime()) && d.getFullYear() > 1990 && d.getFullYear() <= new Date().getFullYear() + 1) return d;
    const yearMatch = dateStr.match(/\b(20\d{2})\b/);
    if (yearMatch) {
        const monthMatch = dateStr.match(/\b(Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec)\w*/i);
        const dayMatch = dateStr.match(/\b(\d{1,2})\b/);
        if (monthMatch) {
            const test = new Date(`${monthMatch[1]} ${dayMatch ? dayMatch[1] : '1'}, ${yearMatch[1]}`);
            if (!isNaN(test.getTime())) return test;
        }
        return new Date(`${yearMatch[1]}-01-01`);
    }
    return null;
}
function renderTrendSVG(months, monthCounts) {
    const W = 900, H = 180;
    const padL = 40, padR = 10, padT = 10, padB = 40;
    const chartW = W - padL - padR;
    const chartH = H - padT - padB;
    const values = months.map(m => monthCounts.get(m) || 0);
    const maxVal = Math.max(...values, 1);
    const points = values.map((v, i) => {
        const x = padL + (i / (months.length - 1)) * chartW;
        const y = padT + chartH - (v / maxVal) * chartH;
        return { x, y, v };
    });
    const fillPath = `M ${points[0].x} ${padT + chartH} ` +
        points.map(p => `L ${p.x} ${p.y}`).join(' ') +
        ` L ${points[points.length - 1].x} ${padT + chartH} Z`;
    const linePath = points.map((p, i) => `${i === 0 ? 'M' : 'L'} ${p.x} ${p.y}`).join(' ');
    let svg = `<div class="trend-chart-container"><svg viewBox="0 0 ${W} ${H}" preserveAspectRatio="none">`;
    const gridSteps = 4;
    for (let i = 0; i <= gridSteps; i++) {
        const y = padT + (i / gridSteps) * chartH;
        const val = Math.round(maxVal * (1 - i / gridSteps));
        svg += `<line x1="${padL}" y1="${y}" x2="${W - padR}" y2="${y}" stroke="#e0e0e0" stroke-width="1"/>`;
        svg += `<text x="${padL - 6}" y="${y + 4}" text-anchor="end" font-size="10" fill="#9aa0a6">${val}</text>`;
    }
    svg += `<path d="${fillPath}" fill="rgba(29,183,167,0.1)"/>`;
    svg += `<path d="${linePath}" fill="none" stroke="#1db7a7" stroke-width="2"/>`;
    points.forEach((p) => {
        if (p.v > 0) {
            svg += `<circle cx="${p.x}" cy="${p.y}" r="3" fill="#1db7a7"/>`;
        }
    });
    months.forEach((m, i) => {
        if (i % 6 === 0 || i === months.length - 1) {
            const x = padL + (i / (months.length - 1)) * chartW;
            const [yr, mo] = m.split('-');
            const label = `${mo}/${yr.slice(2)}`;
            svg += `<text x="${x}" y="${H - 8}" text-anchor="middle" font-size="10" fill="#9aa0a6">${label}</text>`;
        }
    });
    svg += '</svg>';
    svg += '<div style="position:absolute;inset:0;">';
    points.forEach((p, i) => {
        const w = chartW / months.length;
        const [yr, mo] = months[i].split('-');
        svg += `<div style="position:absolute;left:${(p.x - w/2) / W * 100}%;top:0;width:${w / W * 100}%;height:100%;cursor:default;" title="${mo}/${yr}: ${p.v} articles"></div>`;
    });
    svg += '</div></div>';
    return svg;
}
const TAG_CLOUD_STOPWORDS = new Set([
    'the','and','for','are','but','not','you','all','can','had','her','was','one','our','out','has','his',
    'how','its','may','new','now','old','see','way','who','did','get','let','say','she','too','use',
    'that','with','have','this','will','your','from','they','been','call','come','each','make','like',
    'long','look','many','most','over','such','take','than','them','then','very','when','what','some',
    'were','into','more','also','back','only','well','just','about','could','which','their','there',
    'these','other','would','after','being','first','still','where','those','should','while','before',
    'between','through','because','does','here','much','even','both','same','right','under','think',
    'know','help','need','every','might','without','another','using','during','something','really',
    'include','including','often','part','find','provide','want','work','best','good','great','keep',
    'sure','things','used','years','people','time','year','down','made','high','place','small','found',
    'world','going','large','example','number','point','state','given','result','important','possible',
    'different','several','following','also','generally','typically','usually','however','according',
    'based','called','known','specific','type','types','common','well','may','often','around',
    'les','des','est','une','que','dans','pour','pas','sur','plus','avec','sont','qui','par','cette',
    'mais','ses','ont','tout','comme','aussi','bien','fait','nous','vous','elle','aux','ces','leur',
    'entre','sous','chez','sans','peu','peut','fait','dont','encore','alors','depuis','autre','autres'
]);
async function renderTagCloud(containerId, prefix) {
    const container = document.getElementById(containerId);
    if (!container) return;
    container.innerHTML = '<div style="color:var(--text-secondary);font-size:12px;">Loading...</div>';
    const allData = await new Promise(resolve => chrome.storage.local.get(null, resolve));
    const wordFreq = new Map();
    for (const [key, value] of Object.entries(allData)) {
        if (!key.startsWith(prefix) || typeof value !== 'object') continue;
        const fragments = [];
        if (prefix === 'aio_') {
            if (value.additionalTextFragments && Array.isArray(value.additionalTextFragments)) {
                value.additionalTextFragments.forEach(f => {
                    if (f.fragment) fragments.push(f.fragment);
                });
            }
            if (value.textFragmentLinks && Array.isArray(value.textFragmentLinks)) {
                value.textFragmentLinks.forEach(f => {
                    if (f.textFragment) fragments.push(f.textFragment);
                });
            }
            const cits = [...(value.citations || []), ...(value.sidebarCards || [])];
            cits.forEach(c => {
                if (c.textFragment) fragments.push(c.textFragment);
                else if (c.url && c.url.includes('#:~:text=')) {
                    const m = c.url.match(/#:~:text=(.+)/);
                    if (m) { try { fragments.push(decodeURIComponent(m[1])); } catch (e) {} }
                }
            });
        } else {
            if (value.textFragmentLinks && Array.isArray(value.textFragmentLinks)) {
                value.textFragmentLinks.forEach(f => {
                    if (f.textFragment) fragments.push(f.textFragment);
                });
            }
            const cits = value.citations || [];
            cits.forEach(c => {
                if (c.textFragment) fragments.push(c.textFragment);
                else if (c.url && c.url.includes('#:~:text=')) {
                    const m = c.url.match(/#:~:text=(.+)/);
                    if (m) { try { fragments.push(decodeURIComponent(m[1])); } catch (e) {} }
                }
            });
        }
        fragments.forEach(frag => {
            const tokens = frag.toLowerCase().split(/[\s\-\/,.:;!?()[\]{}""''«»„"]+/).filter(t => t.length >= 3);
            tokens.forEach(token => {
                token = token.replace(/^[^a-zA-ZÀ-ÿ]+|[^a-zA-ZÀ-ÿ]+$/g, '');
                if (token.length < 3) return;
                if (TAG_CLOUD_STOPWORDS.has(token)) return;
                if (/^\d+$/.test(token)) return;
                wordFreq.set(token, (wordFreq.get(token) || 0) + 1);
            });
        });
    }
    if (wordFreq.size === 0) {
        container.innerHTML = '<div class="no-data" style="padding:16px;">No text fragments found</div>';
        return;
    }
    const sorted = Array.from(wordFreq.entries()).sort((a, b) => b[1] - a[1]).slice(0, 80);
    const maxFreq = sorted[0][1];
    const minFreq = sorted[sorted.length - 1][1];
    const colors = ['#5c8dca', '#1db7a7', '#e65100', '#8e24aa', '#00695c', '#c62828', '#1565c0', '#ad1457', '#f57f17', '#4527a0'];
    let html = '';
    sorted.forEach(([word, freq], i) => {
        const ratio = maxFreq === minFreq ? 0.5 : (freq - minFreq) / (maxFreq - minFreq);
        const fontSize = Math.round(11 + ratio * 25); // 11px to 36px
        const color = colors[i % colors.length];
        html += `<span class="tag-cloud-word" style="font-size:${fontSize}px;color:${color};" title="${freq} occurrences">${escapeHtml(word)}</span> `;
    });
    container.innerHTML = html;
}
let _aimChartDomainItems = [];
let _aimChartUrlItems = [];
let _aimChartListenersAttached = false;
async function renderAIMGlobalCharts() {
    const data = await computeGlobalDomainData('aim');
    _aimChartDomainItems = classifyAndMerge(data.magiDomains, data.citationDomains, data.intextDomains);
    _aimChartUrlItems = classifyAndMerge(data.magiUrls, data.citationUrls, data.intextUrls);
    renderBarChart(document.getElementById('aimDomainChartContent'), _aimChartDomainItems, 'all', 20, true, 'aimDomains');
    renderBarChart(document.getElementById('aimUrlChartContent'), _aimChartUrlItems, 'all', 20, false, 'aimUrls');
    document.querySelectorAll('#aimDomainChartFilters .chart-filter-btn').forEach(b => b.classList.toggle('active', b.dataset.filter === 'all'));
    document.querySelectorAll('#aimUrlChartFilters .chart-filter-btn').forEach(b => b.classList.toggle('active', b.dataset.filter === 'all'));
    if (_aimChartListenersAttached) return;
    _aimChartListenersAttached = true;
    document.getElementById('aimDomainChartFilters').addEventListener('click', (e) => {
        const btn = e.target.closest('.chart-filter-btn');
        if (!btn) return;
        document.querySelectorAll('#aimDomainChartFilters .chart-filter-btn').forEach(b => b.classList.remove('active'));
        btn.classList.add('active');
        renderBarChart(document.getElementById('aimDomainChartContent'), _aimChartDomainItems, btn.dataset.filter, 20, true, 'aimDomains');
    });
    document.getElementById('aimUrlChartFilters').addEventListener('click', (e) => {
        const btn = e.target.closest('.chart-filter-btn');
        if (!btn) return;
        document.querySelectorAll('#aimUrlChartFilters .chart-filter-btn').forEach(b => b.classList.remove('active'));
        btn.classList.add('active');
        renderBarChart(document.getElementById('aimUrlChartContent'), _aimChartUrlItems, btn.dataset.filter, 20, false, 'aimUrls');
    });
    document.getElementById('aimDomainCopyAllBtn').addEventListener('click', () => {
        showCopyAllUrlsModal(_aimChartUrlItems, 'All AIM URLs — Domains');
    });
    document.getElementById('aimUrlCopyAllBtn').addEventListener('click', () => {
        showCopyAllUrlsModal(_aimChartUrlItems, 'All AIM URLs');
    });
}
async function renderAIMGlobalEntities() {
    const container = document.getElementById('aimEntitiesContent');
    if (!container) return;
    const allData = await new Promise(resolve => chrome.storage.local.get(null, resolve));
    const entityMap = new Map();
    for (const [key, value] of Object.entries(allData)) {
        if (!key.startsWith('aim_') || typeof value !== 'object') continue;
        const q = value.searchQuery || '?';
        if (value.entityLinks && Array.isArray(value.entityLinks)) {
            value.entityLinks.forEach(link => {
                const href = cleanGoogleSearchUrl(link.href || '');
                if (!href) return;
                if (isProductLink(href)) return;
                if (!entityMap.has(href)) {
                    entityMap.set(href, { text: link.text || '', href, queries: [q] });
                } else {
                    const existing = entityMap.get(href);
                    if (!existing.queries.includes(q)) existing.queries.push(q);
                }
            });
        }
    }
    if (entityMap.size === 0) {
        container.innerHTML = '<div class="no-data">No AIM entities found</div>';
        return;
    }
    const entities = Array.from(entityMap.values()).sort((a, b) => b.queries.length - a.queries.length || a.text.localeCompare(b.text));
    let html = `<table class="magi-sources-table"><thead><tr>
        <th style="width:30px;">#</th><th>Entity</th><th style="width:80px;">Occurrences</th><th>Queries</th>
    </tr></thead><tbody>`;
    entities.forEach((ent, i) => {
        html += `<tr>
            <td style="text-align:center;color:var(--text-secondary);font-size:11px;">${i + 1}</td>
            <td><a href="${escapeHtml(ent.href)}" target="_blank" style="color:var(--primary);text-decoration:none;font-weight:500;font-size:12px;" title="${escapeHtml(ent.href)}">${escapeHtml(ent.text)}</a></td>
            <td style="text-align:center;font-weight:600;font-size:12px;">${ent.queries.length}</td>
            <td style="font-size:11px;">${renderQueryLinks(ent.queries)}</td>
        </tr>`;
    });
    html += '</tbody></table>';
    container.innerHTML = html;
    wireQueryNavLinks(container);
    const copyBtn = document.getElementById('aimEntitiesCopyBtn');
    if (copyBtn) {
        copyBtn.onclick = () => {
            const lines = ['#\tEntity\tURL\tOccurrences\tQueries'];
            entities.forEach((ent, i) => {
                lines.push(`${i + 1}\t${ent.text}\t${ent.href}\t${ent.queries.length}\t${ent.queries.join(', ')}`);
            });
            navigator.clipboard.writeText(lines.join('\n')).then(() => {
                copyBtn.textContent = 'Copied!';
                setTimeout(() => { copyBtn.textContent = 'Copy'; }, 1500);
            });
        };
    }
}
async function renderGlobalProducts(containerId, copyBtnId, prefix) {
    const container = document.getElementById(containerId);
    if (!container) return;
    const label = prefix === 'aim_' ? 'AIM' : 'AIO';
    const allData = await new Promise(resolve => chrome.storage.local.get(null, resolve));
    const productMap = new Map();
    for (const [key, value] of Object.entries(allData)) {
        if (!key.startsWith(prefix) || typeof value !== 'object') continue;
        const q = value.searchQuery || '?';
        if (value.entityLinks && Array.isArray(value.entityLinks)) {
            value.entityLinks.forEach(link => {
                const href = cleanGoogleSearchUrl(link.href || '');
                if (!href || !isProductLink(href)) return;
                const name = (link.text || '').trim();
                if (!name) return;
                const key = name.toLowerCase();
                if (!productMap.has(key)) {
                    productMap.set(key, { text: name, href, queries: [q] });
                } else {
                    const existing = productMap.get(key);
                    if (!existing.queries.includes(q)) existing.queries.push(q);
                }
            });
        }
    }
    if (productMap.size === 0) {
        container.innerHTML = `<div class="no-data">No ${label} products found</div>`;
        return;
    }
    const products = Array.from(productMap.values()).sort((a, b) => b.queries.length - a.queries.length || a.text.localeCompare(b.text));
    let html = `<table class="magi-sources-table"><thead><tr>
        <th style="width:30px;">#</th><th>Product</th><th style="width:80px;">Occurrences</th><th>Queries</th>
    </tr></thead><tbody>`;
    products.forEach((p, i) => {
        html += `<tr>
            <td style="text-align:center;color:var(--text-secondary);font-size:11px;">${i + 1}</td>
            <td><a href="${escapeHtml(p.href)}" target="_blank" style="color:var(--primary);text-decoration:none;font-weight:500;font-size:12px;">${escapeHtml(p.text)}</a></td>
            <td style="text-align:center;font-weight:600;font-size:12px;">${p.queries.length}</td>
            <td style="font-size:11px;">${renderQueryLinks(p.queries)}</td>
        </tr>`;
    });
    html += '</tbody></table>';
    container.innerHTML = html;
    wireQueryNavLinks(container);
    const copyBtn = document.getElementById(copyBtnId);
    if (copyBtn) {
        copyBtn.onclick = () => {
            const lines = ['#\tProduct\tURL\tOccurrences\tQueries'];
            products.forEach((p, i) => {
                lines.push(`${i + 1}\t${p.text}\t${p.href}\t${p.queries.length}\t${p.queries.join(', ')}`);
            });
            navigator.clipboard.writeText(lines.join('\n')).then(() => {
                copyBtn.textContent = 'Copied!';
                setTimeout(() => { copyBtn.textContent = 'Copy'; }, 1500);
            });
        };
    }
}
async function renderAIMGlobalPlaces() {
    const container = document.getElementById('aimPlacesGlobalContent');
    if (!container) return;
    const allData = await new Promise(resolve => chrome.storage.local.get(null, resolve));
    const placeMap = new Map();
    for (const [key, value] of Object.entries(allData)) {
        if (!key.startsWith('aim_') || typeof value !== 'object') continue;
        const q = value.searchQuery || '?';
        if (value.placeEntities && Array.isArray(value.placeEntities)) {
            value.placeEntities.forEach(place => {
                const mid = place.mid || '';
                const dedupKey = mid || (place.name || '').toLowerCase();
                if (!dedupKey) return;
                if (!placeMap.has(dedupKey)) {
                    placeMap.set(dedupKey, { name: place.name || '?', mid, type: place.type || 'place', href: place.href || '', queries: [q] });
                } else {
                    const existing = placeMap.get(dedupKey);
                    if (!existing.queries.includes(q)) existing.queries.push(q);
                }
            });
        }
    }
    if (placeMap.size === 0) {
        container.innerHTML = '<div class="no-data">No AIM places found</div>';
        return;
    }
    const places = Array.from(placeMap.values()).sort((a, b) => b.queries.length - a.queries.length || a.name.localeCompare(b.name));
    let html = `<table class="magi-sources-table"><thead><tr>
        <th style="width:30px;">#</th><th>Name</th><th style="width:120px;">MID</th><th style="width:80px;">Type</th><th style="width:80px;">Occurrences</th><th>Queries</th>
    </tr></thead><tbody>`;
    places.forEach((p, i) => {
        html += `<tr>
            <td style="text-align:center;color:var(--text-secondary);font-size:11px;">${i + 1}</td>
            <td>${p.href ? `<a href="${escapeHtml(p.href)}" target="_blank" style="color:var(--primary);text-decoration:none;font-weight:500;font-size:12px;">${escapeHtml(p.name)}</a>` : escapeHtml(p.name)}</td>
            <td style="font-size:11px;color:var(--text-secondary);font-family:var(--font-mono);">${escapeHtml(p.mid)}</td>
            <td style="text-align:center;"><span style="padding:2px 8px;border-radius:4px;font-size:10px;font-weight:600;background:#e3f0fa;color:#1565c0;">${escapeHtml(p.type)}</span></td>
            <td style="text-align:center;font-weight:600;font-size:12px;">${p.queries.length}</td>
            <td style="font-size:11px;">${renderQueryLinks(p.queries)}</td>
        </tr>`;
    });
    html += '</tbody></table>';
    container.innerHTML = html;
    wireQueryNavLinks(container);
    const copyBtn = document.getElementById('aimPlacesCopyBtn');
    if (copyBtn) {
        copyBtn.onclick = () => {
            const lines = ['#\tName\tMID\tType\tOccurrences\tQueries'];
            places.forEach((p, i) => {
                lines.push(`${i + 1}\t${p.name}\t${p.mid}\t${p.type}\t${p.queries.length}\t${p.queries.join(', ')}`);
            });
            navigator.clipboard.writeText(lines.join('\n')).then(() => {
                copyBtn.textContent = 'Copied!';
                setTimeout(() => { copyBtn.textContent = 'Copy'; }, 1500);
            });
        };
    }
}
async function renderAIMGlobalKG() {
    const container = document.getElementById('aimKGGlobalContent');
    if (!container) return;
    const allData = await new Promise(resolve => chrome.storage.local.get(null, resolve));
    const kgMap = new Map();
    for (const [key, value] of Object.entries(allData)) {
        if (!key.startsWith('aim_') || typeof value !== 'object') continue;
        const q = value.searchQuery || '?';
        if (value.kgEntities && Array.isArray(value.kgEntities)) {
            value.kgEntities.forEach(ent => {
                const kgmid = ent.kgmid || '';
                const dedupKey = kgmid || (ent.name || '').toLowerCase();
                if (!dedupKey) return;
                if (!kgMap.has(dedupKey)) {
                    let displayName = ent.name || '?';
                    if ((!displayName || displayName === '?') && ent.href) {
                        try { const u = new URL(ent.href); const qp = u.searchParams.get('q'); if (qp) displayName = qp.replace(/\+/g, ' '); } catch(e) {}
                    }
                    kgMap.set(dedupKey, { name: displayName, kgmid, href: ent.href || '', queries: [q] });
                } else {
                    const existing = kgMap.get(dedupKey);
                    if (!existing.queries.includes(q)) existing.queries.push(q);
                }
            });
        }
    }
    if (kgMap.size === 0) {
        container.innerHTML = '<div class="no-data">No AIM Knowledge Graph entities found</div>';
        return;
    }
    const kgEntities = Array.from(kgMap.values()).sort((a, b) => b.queries.length - a.queries.length || a.name.localeCompare(b.name));
    let html = `<table class="magi-sources-table"><thead><tr>
        <th style="width:30px;">#</th><th>Name</th><th style="width:140px;">KGMID</th><th style="width:80px;">Occurrences</th><th>Queries</th>
    </tr></thead><tbody>`;
    kgEntities.forEach((ent, i) => {
        html += `<tr>
            <td style="text-align:center;color:var(--text-secondary);font-size:11px;">${i + 1}</td>
            <td>${ent.href ? `<a href="${escapeHtml(ent.href)}" target="_blank" style="color:var(--primary);text-decoration:none;font-weight:500;font-size:12px;">${escapeHtml(ent.name)}</a>` : escapeHtml(ent.name)}</td>
            <td style="font-size:11px;color:var(--text-secondary);font-family:var(--font-mono);">${escapeHtml(ent.kgmid)}</td>
            <td style="text-align:center;font-weight:600;font-size:12px;">${ent.queries.length}</td>
            <td style="font-size:11px;">${renderQueryLinks(ent.queries)}</td>
        </tr>`;
    });
    html += '</tbody></table>';
    container.innerHTML = html;
    wireQueryNavLinks(container);
    const copyBtn = document.getElementById('aimKGCopyBtn');
    if (copyBtn) {
        copyBtn.onclick = () => {
            const lines = ['#\tName\tKGMID\tOccurrences\tQueries'];
            kgEntities.forEach((ent, i) => {
                lines.push(`${i + 1}\t${ent.name}\t${ent.kgmid}\t${ent.queries.length}\t${ent.queries.join(', ')}`);
            });
            navigator.clipboard.writeText(lines.join('\n')).then(() => {
                copyBtn.textContent = 'Copied!';
                setTimeout(() => { copyBtn.textContent = 'Copy'; }, 1500);
            });
        };
    }
}
let currentQueriesList = [];
let filteredQueriesList = [];
async function showAllQueriesView() {
    const allData = await getAllStorageData('all');
    const allQueries = processStorageData(allData, 'all');
    const sortedQueries = [...allQueries].sort((a, b) => {
        const aTime = Math.max(...(a.captures?.map(c => new Date(c.timestamp).getTime()).filter(t => !isNaN(t)) || [0]));
        const bTime = Math.max(...(b.captures?.map(c => new Date(c.timestamp).getTime()).filter(t => !isNaN(t)) || [0]));
        return bTime - aTime;
    });
    currentQueriesList = sortedQueries;
    filteredQueriesList = [...sortedQueries];
    document.getElementById('dataContainer').style.display = 'none';
    document.getElementById('noDataContainer').style.display = 'none';
    document.getElementById('allDataAttridContainer').style.display = 'none';
    document.getElementById('allQueriesContainer').style.display = 'block';
    document.getElementById('showAllBtn').style.display = 'none';
    document.getElementById('showDataAttridBtn').style.display = 'block';
    document.getElementById('showSingleBtn').style.display = 'block';
    document.getElementById('querySelect').style.display = 'none';
    const sortSelect = document.getElementById('sortSelect');
    if (sortSelect) sortSelect.value = 'chronological';
    setupAllQueriesEventListeners();
    await renderAllQueries();
}
async function showSingleQueryView() {
    document.getElementById('allQueriesContainer').style.display = 'none';
    document.getElementById('allDataAttridContainer').style.display = 'none';
    document.getElementById('showAllBtn').style.display = 'block';
    document.getElementById('showDataAttridBtn').style.display = 'block';
    document.getElementById('showSingleBtn').style.display = 'none';
    const selectEl = document.getElementById('querySelect');
    selectEl.style.display = 'block';
    if (_currentQueries.length > 0 && !selectEl.value) {
        selectEl.value = '0';
        selectEl.dispatchEvent(new Event('change'));
    } else if (selectEl.value) {
        document.getElementById('dataContainer').style.display = 'block';
        document.getElementById('noDataContainer').style.display = 'none';
    } else {
        document.getElementById('dataContainer').style.display = 'none';
        document.getElementById('noDataContainer').style.display = 'block';
    }
}
async function showAllDataAttridView() {
    document.getElementById('dataContainer').style.display = 'none';
    document.getElementById('noDataContainer').style.display = 'none';
    document.getElementById('allQueriesContainer').style.display = 'none';
    document.getElementById('allDataAttridContainer').style.display = 'block';
    document.getElementById('showAllBtn').style.display = 'block';
    document.getElementById('showDataAttridBtn').style.display = 'none';
    document.getElementById('showSingleBtn').style.display = 'block';
    document.getElementById('querySelect').style.display = 'none';
    styleGlobalTab(document.getElementById('gstAIO'), globalSubTab === 'aio');
    styleGlobalTab(document.getElementById('gstAIM'), globalSubTab === 'aim');
    document.getElementById('globalAIOContent').style.display = globalSubTab === 'aio' ? 'block' : 'none';
    document.getElementById('globalAIMContent').style.display = globalSubTab === 'aim' ? 'block' : 'none';
    if (globalSubTab === 'aio') {
        await renderGlobalAIOSubTab();
    } else {
        await renderGlobalAIMSubTab();
    }
}
async function renderGlobalAIOSubTab() {
    await renderTagCloud('aioTagCloudContent', 'aio_');
    await renderGlobalCharts();
    await renderAllAIOEntities();
    await renderGlobalProducts('aioProductsContent', 'aioProductsCopyBtn', 'aio_');
    await renderAIDomains();
}
async function renderGlobalAIMSubTab() {
    await renderTagCloud('aimTagCloudContent', 'aim_');
    await renderFreshness('aimFreshnessContent', 'aim_');
    await renderAIMGlobalCharts();
    await renderAIMGlobalEntities();
    await renderGlobalProducts('aimProductsContent', 'aimProductsCopyBtn', 'aim_');
    await renderAIMGlobalPlaces();
    await renderAIMGlobalKG();
}
function styleGlobalTab(el, active) {
    el.style.background = active ? 'var(--primary)' : 'var(--card)';
    el.style.color = active ? 'white' : 'var(--text-secondary)';
    el.style.borderColor = active ? 'var(--primary)' : 'var(--border)';
}
function switchGlobalSubTab(newTab) {
    globalSubTab = newTab;
    styleGlobalTab(document.getElementById('gstAIO'), newTab === 'aio');
    styleGlobalTab(document.getElementById('gstAIM'), newTab === 'aim');
    document.getElementById('globalAIOContent').style.display = newTab === 'aio' ? 'block' : 'none';
    document.getElementById('globalAIMContent').style.display = newTab === 'aim' ? 'block' : 'none';
    if (newTab === 'aio') {
        renderGlobalAIOSubTab();
    } else {
        renderGlobalAIMSubTab();
    }
}
async function renderAllQueries() {
    const container = document.getElementById('allQueriesList');
    container.innerHTML = '';
    if (filteredQueriesList.length === 0) {
        container.innerHTML = '<div class="no-data">No queries found</div>';
        return;
    }
    for (const query of filteredQueriesList) {
        const queryCard = createQueryCard(query);
        container.appendChild(queryCard);
    }
}
function createQueryCard(query) {
    const card = document.createElement('div');
    card.className = 'query-item-card';
    const capture = query.captures[0] || {};
    const isAIMCapture = capture.captureType === 'aim';
    const citCount = (capture.citations || []).length;
    const sidebarCount = isAIMCapture ? (capture.sidebarSources || []).length : (capture.sidebarCards || []).length;
    const statusValue = isAIMCapture ? (capture.aimStatus || 'absent') : (capture.aioStatus || 'absent');
    const statusLabel = isAIMCapture ? 'AIM' : 'AIO';
    const entityCount = (capture.entityLinks || []).length;
    const magiCount = (capture.magiInnerHTML || []).length;
    const lastUpdated = capture.timestamp ? new Date(capture.timestamp) : null;
    const header = document.createElement('div');
    header.className = 'query-item-header';
    const titleDiv = document.createElement('div');
    const title = document.createElement('div');
    title.className = 'query-item-title';
    title.textContent = query.query;
    titleDiv.appendChild(title);
    const meta = document.createElement('div');
    meta.className = 'query-item-meta';
    const inTextCount = (capture.externalLinks || []).length;
    let extraInfo = '';
    if (isAIMCapture) {
        if (entityCount > 0) extraInfo += `<span>${entityCount} entities</span>`;
        if (inTextCount > 0) extraInfo += `<span>${inTextCount} in-text links</span>`;
    } else {
        if (magiCount > 0) extraInfo += `<span>${magiCount} grounding</span>`;
        if (entityCount > 0) extraInfo += `<span>${entityCount} entities</span>`;
    }
    meta.innerHTML = `
        <span>${lastUpdated ? getTimeAgo(lastUpdated) : 'Unknown'}</span>
        <span style="padding:1px 6px;border-radius:8px;font-size:10px;font-weight:600;${isAIMCapture ? 'background:#f3e5f5;color:#8e24aa;' : 'background:#e8f0fe;color:#1a73e8;'}">${statusLabel}</span>
        <span>${statusLabel}: ${escapeHtml(statusValue)}</span>
        <span>${citCount} pool · ${sidebarCount} sidebar</span>
        ${extraInfo}
    `;
    titleDiv.appendChild(meta);
    header.appendChild(titleDiv);
    const actionsDiv = document.createElement('div');
    actionsDiv.className = 'query-item-actions';
    const deleteBtn = document.createElement('button');
    deleteBtn.className = 'delete-query-btn';
    deleteBtn.textContent = '🗑️ Delete';
    deleteBtn.addEventListener('click', async (e) => {
        e.stopPropagation();
        await deleteQuery(query.query);
    });
    actionsDiv.appendChild(deleteBtn);
    header.appendChild(actionsDiv);
    card.appendChild(header);
    card.addEventListener('click', async () => {
        document.getElementById('allQueriesContainer').style.display = 'none';
        document.getElementById('dataContainer').style.display = 'block';
        document.getElementById('showAllBtn').style.display = 'block';
        document.getElementById('showSingleBtn').style.display = 'none';
        await renderQueryDetail(query);
    });
    return card;
}
function setupAllQueriesEventListeners() {
    const searchInput = document.getElementById('querySearchInput');
    const sortSelect = document.getElementById('sortSelect');
    const exportBtn = document.getElementById('exportAllQueriesBtn');
    const deleteBtn = document.getElementById('deleteAllQueriesBtn');
    searchInput.removeEventListener('input', handleQuerySearch);
    searchInput.addEventListener('input', handleQuerySearch);
    sortSelect.removeEventListener('change', handleSortChange);
    sortSelect.addEventListener('change', handleSortChange);
    if (exportBtn) {
        exportBtn.removeEventListener('click', exportAllQueries);
        exportBtn.addEventListener('click', async () => {
            await exportAllQueries();
        });
    }
    if (deleteBtn) {
        deleteBtn.removeEventListener('click', deleteAllQueries);
        deleteBtn.addEventListener('click', async () => {
            await deleteAllQueries();
        });
    }
}
async function handleQuerySearch(e) {
    const searchTerm = e.target.value.toLowerCase().trim();
    if (searchTerm === '') {
        filteredQueriesList = [...currentQueriesList];
    } else {
        filteredQueriesList = currentQueriesList.filter(query =>
            query.query.toLowerCase().includes(searchTerm)
        );
    }
    const sortSelect = document.getElementById('sortSelect');
    if (sortSelect) {
        const sortType = sortSelect.value;
        filteredQueriesList.sort((a, b) => {
            switch (sortType) {
                case 'alphabetical':
                    return a.query.localeCompare(b.query);
                case 'chronological':
                    const aTime = Math.max(...(a.captures.map(c => new Date(c.timestamp).getTime()).filter(t => !isNaN(t))));
                    const bTime = Math.max(...(b.captures.map(c => new Date(c.timestamp).getTime()).filter(t => !isNaN(t))));
                    return bTime - aTime;
                case 'captures':
                    return b.captures.length - a.captures.length;
                default:
                    return 0;
            }
        });
    }
    await renderAllQueries();
}
async function handleSortChange(e) {
    const sortType = e.target.value;
    filteredQueriesList.sort((a, b) => {
        switch (sortType) {
            case 'alphabetical':
                return a.query.localeCompare(b.query);
            case 'chronological':
                const aTime = Math.max(...(a.captures.map(c => new Date(c.timestamp).getTime()).filter(t => !isNaN(t))));
                const bTime = Math.max(...(b.captures.map(c => new Date(c.timestamp).getTime()).filter(t => !isNaN(t))));
                return bTime - aTime;
            case 'captures':
                return b.captures.length - a.captures.length;
            default:
                return 0;
        }
    });
    await renderAllQueries();
}
async function deleteQuery(queryText) {
    if (!confirm(`Are you sure you want to delete the query "${queryText}" and all its data?`)) {
        return;
    }
    try {
        const data = await new Promise((resolve) => {
            chrome.storage.local.get(null, resolve);
        });
        const keysToRemove = [];
        for (const [key, value] of Object.entries(data)) {
            if ((key.startsWith('aio_') || key.startsWith('aim_')) && value && value.searchQuery === queryText) {
                keysToRemove.push(key);
            }
        }
        if (keysToRemove.length > 0) {
            await new Promise((resolve) => {
                chrome.storage.local.remove(keysToRemove, resolve);
            });
        }
        currentQueriesList = currentQueriesList.filter(q => q.query !== queryText);
        filteredQueriesList = filteredQueriesList.filter(q => q.query !== queryText);
        await renderAllQueries();
        alert(`Query "${queryText}" successfully deleted (${keysToRemove.length} item(s) removed)`);
    } catch (error) {
        alert(`Deletion error: ${error.message}`);
    }
}
async function deleteAllQueries() {
    if (!confirm('Are you sure you want to delete ALL queries and their data?\n\nThis action is irreversible!')) {
        return;
    }
    if (!confirm('Are you absolutely sure? All data will be lost.')) {
        return;
    }
    try {
        const data = await new Promise((resolve) => {
            chrome.storage.local.get(null, resolve);
        });
        const keysToRemove = Object.keys(data).filter(key => key.startsWith('aio_') || key.startsWith('aim_'));
        if (keysToRemove.length > 0) {
            await new Promise((resolve) => {
                chrome.storage.local.remove(keysToRemove, resolve);
            });
        }
        currentQueriesList = [];
        filteredQueriesList = [];
        _currentQueries = await loadQueries();
        await renderAllQueries();
        alert(`All data has been deleted (${keysToRemove.length} item(s) removed)`);
    } catch (error) {
        alert(`Deletion error: ${error.message}`);
    }
}
async function exportAllDataGlobal() {
    const statusEl = document.getElementById('exportAllBtn');
    const originalText = statusEl ? statusEl.textContent : '';
    try {
        if (statusEl) statusEl.textContent = '⏳ Export...';
        const allData = await new Promise(resolve => chrome.storage.local.get(null, resolve));
        const captures = [];
        const exportPrefixes = currentDataType === 'aim' ? ['aim_'] : currentDataType === 'cross' ? ['aio_', 'aim_'] : ['aio_'];
        for (const [key, value] of Object.entries(allData)) {
            if (exportPrefixes.some(p => key.startsWith(p)) && typeof value === 'object') {
                captures.push({ ...value, _type: key.startsWith('aim_') ? 'aim' : 'aio' });
            }
        }
        const exportPayload = {
            exportDate: new Date().toISOString(),
            version: '1.0',
            dataType: currentDataType,
            totalCaptures: captures.length,
            captures: captures
        };
        const output = JSON.stringify(exportPayload, null, 2);
        const blob = new Blob([output], { type: 'application/json' });
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `aio-inspector-export-${new Date().toISOString().split('T')[0]}.json`;
        a.click();
        URL.revokeObjectURL(url);
        if (statusEl) statusEl.textContent = originalText;
        alert(`Export successful: ${captures.length} capture(s)`);
    } catch (error) {
        if (statusEl) statusEl.textContent = originalText;
        alert(`Export error: ${error.message}`);
    }
}
async function exportAllQueries() {
    try {
        const exportData = [];
        for (const query of currentQueriesList) {
            const capture = query.captures[0] || {};
            exportData.push({
                query: query.query,
                aioStatus: capture.aioStatus || 'absent',
                citationCount: (capture.citations || []).length,
                sidebarCount: (capture.sidebarCards || []).length,
                entityCount: (capture.entityLinks || []).length,
                magiCount: (capture.magiInnerHTML || []).length,
                lastUpdated: capture.timestamp || ''
            });
        }
        let csvContent = 'Query,AIO Status,Citations (pool),Sidebar,Entities,Grounding URLs,Last Updated\n';
        exportData.forEach(item => {
            const lastUpdate = item.lastUpdated ? new Date(item.lastUpdated).toLocaleString('en-US') : 'N/A';
            csvContent += `"${item.query.replace(/"/g, '""')}","${item.aioStatus}",${item.citationCount},${item.sidebarCount},${item.entityCount},${item.magiCount},"${lastUpdate}"\n`;
        });
        const BOM = '\uFEFF';
        const blob = new Blob([BOM + csvContent], { type: 'text/csv;charset=utf-8;' });
        const url = URL.createObjectURL(blob);
        const link = document.createElement('a');
        link.href = url;
        link.download = `aio-inspector-queries-${new Date().toISOString().split('T')[0]}.csv`;
        link.click();
        URL.revokeObjectURL(url);
        alert(`Export successful: ${exportData.length} query(ies) exported`);
    } catch (error) {
        alert(`Export error: ${error.message}`);
    }
}
async function setupToggleCaptureButton() {
    const toggleBtn = document.getElementById('toggleCaptureBtn');
    async function updateButtonState() {
        const result = await chrome.storage.local.get(['captureEnabled']);
        const isEnabled = result.captureEnabled !== false;
        updateToggleButton(toggleBtn, isEnabled);
        return isEnabled;
    }
    await updateButtonState();
    toggleBtn.addEventListener('click', async () => {
        const currentState = await updateButtonState();
        const newState = !currentState;
        await chrome.storage.local.set({ captureEnabled: newState });
        updateToggleButton(toggleBtn, newState);
        showNotification(newState ? 'Capture enabled' : 'Capture paused');
    });
}
function updateToggleButton(button, isEnabled) {
    if (isEnabled) {
        button.innerHTML = '⏸️ Pause capture';
        button.style.background = 'var(--warning)';
        button.style.color = 'white';
    } else {
        button.innerHTML = '▶️ Enable capture';
        button.style.background = 'var(--success)';
        button.style.color = 'white';
    }
}
function showNotification(message) {
    const notif = document.createElement('div');
    notif.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: var(--resoneo-dark-blue);
        color: white;
        padding: 16px 24px;
        border-radius: 8px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.3);
        z-index: 10000;
        font-weight: 600;
    `;
    notif.textContent = message;
    document.body.appendChild(notif);
    setTimeout(() => {
        notif.style.transition = 'opacity 0.3s';
        notif.style.opacity = '0';
        setTimeout(() => notif.remove(), 300);
    }, 2000);
}
async function renderCrossComparison(queryText) {
    const section = document.getElementById('crossComparisonSection');
    const content = document.getElementById('crossComparisonContent');
    if (!section || !content) return;
    const allData = await new Promise(resolve => chrome.storage.local.get(null, resolve));
    let aioCapture = null, aimCapture = null;
    for (const [key, value] of Object.entries(allData)) {
        if (key.startsWith('aio_') && typeof value === 'object' && value.searchQuery === queryText) {
            aioCapture = value;
        }
        if (key.startsWith('aim_') && typeof value === 'object' && value.searchQuery === queryText) {
            aimCapture = value;
        }
    }
    if (!aioCapture || !aimCapture) {
        section.style.display = 'none';
        content.innerHTML = '<div class="no-data">Need both AIO and AIM captures for the same query to compare</div>';
        return;
    }
    section.style.display = 'block';
    const contextCardContainer = document.getElementById('contextCard');
    if (contextCardContainer) {
        aioCapture.captureType = 'aio';
        aimCapture.captureType = 'aim';
        contextCardContainer.innerHTML = `
            <div class="card" style="background: var(--card); margin-bottom: 16px;">
                <div style="margin-bottom: 12px;">
                    <div style="font-family: var(--font-heading); font-size: 24px; font-weight: 700; color: var(--text-primary); margin-bottom: 16px;">
                        ${escapeHtml(queryText)}
                    </div>
                </div>
                ${buildContextChips(aioCapture, 'AIO')}
                <div style="margin-top:12px;"></div>
                ${buildContextChips(aimCapture, 'AI Mode')}
            </div>
        `;
    }
    const urlInfo = new Map(); // normUrl → { url, domain, inAIOGrounding, inAIMGrounding, inAIOCit, inAIMCit, inAIOSidebar, inAIMSidebar }
    function trackUrl(url, field) {
        if (!url || isExcludedUrl(url)) return;
        const norm = normalizeUrlForComparison(url, false);
        if (!norm) return;
        if (!urlInfo.has(norm)) {
            let domain = '';
            try { domain = new URL(url).hostname.replace('www.', ''); } catch (e) {}
            urlInfo.set(norm, { url, domain, inAIOGrounding: false, inAIMGrounding: false, inAIOCit: false, inAIMCit: false, inAIOSidebar: false, inAIMSidebar: false });
        }
        urlInfo.get(norm)[field] = true;
    }
    (aioCapture.magiInnerHTML || []).forEach(m => trackUrl(m.url, 'inAIOGrounding'));
    (aioCapture.citations || []).forEach(c => trackUrl(c.url, 'inAIOCit'));
    (aioCapture.sidebarCards || []).forEach(c => trackUrl(c.url, 'inAIOSidebar'));
    (aimCapture.magiInnerHTML || []).forEach(m => trackUrl(m.url, 'inAIMGrounding'));
    (aimCapture.citations || []).forEach(c => trackUrl(c.url, 'inAIMCit'));
    (aimCapture.sidebarSources || []).forEach(c => trackUrl(c.url, 'inAIMSidebar'));
    const allEntries = [...urlInfo.entries()].map(([norm, info]) => {
        const inAIO = info.inAIOGrounding || info.inAIOCit || info.inAIOSidebar;
        const inAIM = info.inAIMGrounding || info.inAIMCit || info.inAIMSidebar;
        const isGroundingOnly = (info.inAIOGrounding || info.inAIMGrounding) && !info.inAIOCit && !info.inAIMCit && !info.inAIOSidebar && !info.inAIMSidebar;
        return { norm, ...info, inAIO, inAIM, isGroundingOnly };
    });
    const aioOnly = allEntries.filter(e => e.inAIO && !e.inAIM);
    const aimOnly = allEntries.filter(e => !e.inAIO && e.inAIM);
    const shared = allEntries.filter(e => e.inAIO && e.inAIM);
    let hideGrounding = false;
    let activeVennZone = null; // 'aio', 'aim', 'shared', or null
    let sortCol = 'domain'; // 'aio', 'domain', 'aim'
    let sortAsc = true;
    function renderCross() {
        const filteredEntries = hideGrounding ? allEntries.filter(e => !e.isGroundingOnly) : allEntries;
        const fAioOnly = filteredEntries.filter(e => e.inAIO && !e.inAIM);
        const fAimOnly = filteredEntries.filter(e => !e.inAIO && e.inAIM);
        const fShared = filteredEntries.filter(e => e.inAIO && e.inAIM);
        let html = `<div style="display:flex;flex-direction:column;align-items:center;margin-bottom:20px;">`;
        const vW = 360, vH = 200, r = 80;
        const aioX = vW / 2 - 40, aimX = vW / 2 + 40, cY = vH / 2;
        html += `<svg width="${vW}" height="${vH}" viewBox="0 0 ${vW} ${vH}">`;
        html += `<circle cx="${aioX}" cy="${cY}" r="${r}" fill="${activeVennZone === 'aio' ? 'rgba(26,115,232,0.35)' : 'rgba(26,115,232,0.15)'}" stroke="#1a73e8" stroke-width="2" style="cursor:pointer;" data-zone="aio"/>`;
        html += `<circle cx="${aimX}" cy="${cY}" r="${r}" fill="${activeVennZone === 'aim' ? 'rgba(142,36,170,0.35)' : 'rgba(142,36,170,0.15)'}" stroke="#8e24aa" stroke-width="2" style="cursor:pointer;" data-zone="aim"/>`;
        html += `<path d="${describeIntersection(aioX, aimX, cY, r)}" fill="${activeVennZone === 'shared' ? 'rgba(100,100,100,0.4)' : 'rgba(100,100,100,0.2)'}" style="cursor:pointer;" data-zone="shared"/>`;
        html += `<text x="${aioX - 40}" y="${cY - 10}" text-anchor="middle" font-size="13" font-weight="700" fill="#1a73e8" style="pointer-events:none;">AIO</text>`;
        html += `<text x="${aioX - 40}" y="${cY + 10}" text-anchor="middle" font-size="20" font-weight="700" fill="#1a73e8" style="pointer-events:none;">${fAioOnly.length}</text>`;
        html += `<text x="${aimX + 40}" y="${cY - 10}" text-anchor="middle" font-size="13" font-weight="700" fill="#8e24aa" style="pointer-events:none;">AIM</text>`;
        html += `<text x="${aimX + 40}" y="${cY + 10}" text-anchor="middle" font-size="20" font-weight="700" fill="#8e24aa" style="pointer-events:none;">${fAimOnly.length}</text>`;
        html += `<text x="${(aioX + aimX) / 2}" y="${cY - 10}" text-anchor="middle" font-size="11" font-weight="600" fill="#333" style="pointer-events:none;">Both</text>`;
        html += `<text x="${(aioX + aimX) / 2}" y="${cY + 12}" text-anchor="middle" font-size="20" font-weight="700" fill="#333" style="pointer-events:none;">${fShared.length}</text>`;
        html += `</svg>`;
        html += `<div style="display:flex;align-items:center;gap:16px;margin-top:8px;">`;
        html += `<div style="font-size:12px;color:var(--text-secondary);">Total: <strong>${filteredEntries.length}</strong> URLs</div>`;
        html += `<label style="font-size:12px;color:var(--text-secondary);cursor:pointer;display:flex;align-items:center;gap:6px;">
            <input type="checkbox" id="crossShowBoth" ${activeVennZone === 'shared' ? 'checked' : ''} style="cursor:pointer;"> Show only shared
        </label>`;
        html += `<label style="font-size:12px;color:var(--text-secondary);cursor:pointer;display:flex;align-items:center;gap:6px;">
            <input type="checkbox" id="crossHideGrounding" ${hideGrounding ? 'checked' : ''} style="cursor:pointer;"> Hide grounding-only URLs
        </label>`;
        html += `</div></div>`;
        let itemsToShow = filteredEntries;
        if (activeVennZone === 'aio') itemsToShow = fAioOnly;
        else if (activeVennZone === 'aim') itemsToShow = fAimOnly;
        else if (activeVennZone === 'shared') itemsToShow = fShared;
        const sortArrow = (col) => sortCol === col ? (sortAsc ? ' &#x25B2;' : ' &#x25BC;') : '';
        const sorted = [...itemsToShow].sort((a, b) => {
            let cmp = 0;
            if (sortCol === 'aio') cmp = (a.inAIO ? 1 : 0) - (b.inAIO ? 1 : 0);
            else if (sortCol === 'aim') cmp = (a.inAIM ? 1 : 0) - (b.inAIM ? 1 : 0);
            else cmp = (a.domain || '').localeCompare(b.domain || '');
            return sortAsc ? cmp : -cmp;
        });
        if (sorted.length > 0) {
            const yStyle = Y_STYLE;
            const nStyle = N_STYLE;
            const thStyle = 'cursor:pointer;user-select:none;';
            html += `<table class="magi-sources-table" style="margin-top:16px;"><thead><tr>
                <th style="width:60px;${thStyle}" data-sort="aio">AIO${sortArrow('aio')}</th>
                <th style="${thStyle}" data-sort="domain">URL${sortArrow('domain')}</th>
                <th style="width:60px;${thStyle}" data-sort="aim">AIM${sortArrow('aim')}</th>
            </tr></thead><tbody>`;
            sorted.forEach((item) => {
                const urlCell = `${faviconImg(item.domain)}<a href="${escapeHtml(item.url)}" target="_blank" style="color:var(--primary);text-decoration:none;word-break:break-all;">${escapeHtml(item.url)}</a>`;
                html += `<tr>
                    <td style="text-align:center;${item.inAIO ? yStyle : nStyle}">${item.inAIO ? '&#x2713;' : ''}</td>
                    <td style="font-size:11px;">${urlCell}</td>
                    <td style="text-align:center;${item.inAIM ? yStyle : nStyle}">${item.inAIM ? '&#x2713;' : ''}</td>
                </tr>`;
            });
            html += '</tbody></table>';
            html += `<button id="crossCopyBtn" class="copy-all-urls-btn" style="margin-top:8px;">Copy</button>`;
        }
        content.innerHTML = html;
        content.querySelectorAll('[data-zone]').forEach(el => {
            el.addEventListener('click', () => {
                const zone = el.dataset.zone;
                activeVennZone = activeVennZone === zone ? null : zone;
                renderCross();
            });
        });
        content.querySelectorAll('[data-sort]').forEach(th => {
            th.addEventListener('click', () => {
                const col = th.dataset.sort;
                if (sortCol === col) { sortAsc = !sortAsc; } else { sortCol = col; sortAsc = true; }
                renderCross();
            });
        });
        const bothChk = content.querySelector('#crossShowBoth');
        if (bothChk) {
            bothChk.addEventListener('change', () => {
                activeVennZone = bothChk.checked ? 'shared' : null;
                renderCross();
            });
        }
        const chk = content.querySelector('#crossHideGrounding');
        if (chk) {
            chk.addEventListener('change', () => {
                hideGrounding = chk.checked;
                renderCross();
            });
        }
        const copyBtn = content.querySelector('#crossCopyBtn');
        if (copyBtn) {
            copyBtn.addEventListener('click', () => {
                const lines = ['AIO\tURL\tAIM'];
                sorted.forEach(item => {
                    lines.push(`${item.inAIO ? 'y' : 'n'}\t${item.url}\t${item.inAIM ? 'y' : 'n'}`);
                });
                navigator.clipboard.writeText(lines.join('\n')).then(() => {
                    copyBtn.textContent = 'Copied!';
                    setTimeout(() => { copyBtn.textContent = 'Copy'; }, 1500);
                });
            });
        }
    }
    renderCross();
}
function describeIntersection(x1, x2, cy, r) {
    const d = x2 - x1;
    if (d >= 2 * r) return '';
    const halfD = d / 2;
    const h = Math.sqrt(r * r - halfD * halfD);
    const ix = (x1 + x2) / 2;
    const iy1 = cy - h;
    const iy2 = cy + h;
    return `M ${ix} ${iy1} A ${r} ${r} 0 0 1 ${ix} ${iy2} A ${r} ${r} 0 0 0 ${ix} ${iy1} Z`;
}
async function switchDataType(newType) {
    const selectEl = document.getElementById('querySelect');
    const prevIndex = selectEl.value ? parseInt(selectEl.value) : -1;
    const prevQuery = (prevIndex >= 0 && _currentQueries[prevIndex]) ? _currentQueries[prevIndex].query : null;
    currentDataType = newType;
    document.getElementById('dtAIO').classList.toggle('primary', newType === 'aio');
    document.getElementById('dtAIM').classList.toggle('primary', newType === 'aim');
    document.getElementById('dtCross').classList.toggle('primary', newType === 'cross');
    document.getElementById('dataContainer').style.display = 'none';
    document.getElementById('noDataContainer').style.display = 'block';
    document.getElementById('allQueriesContainer').style.display = 'none';
    document.getElementById('allDataAttridContainer').style.display = 'none';
    document.getElementById('showAllBtn').style.display = 'block';
    document.getElementById('showDataAttridBtn').style.display = 'block';
    document.getElementById('showSingleBtn').style.display = 'none';
    document.getElementById('querySelect').style.display = 'block';
    selectEl.value = '';
    _currentQueries = await loadQueries();
    let targetIndex = -1;
    if (prevQuery) {
        targetIndex = _currentQueries.findIndex(q => q.query === prevQuery);
    }
    if (targetIndex < 0 && _currentQueries.length > 0) {
        targetIndex = 0; // Default to first (most recent) query
    }
    if (targetIndex >= 0) {
        selectEl.value = targetIndex.toString();
        selectEl.dispatchEvent(new Event('change'));
    }
}
let _currentQueries = [];
document.addEventListener('DOMContentLoaded', async () => {
    showLoading(true);
    const urlParams = new URLSearchParams(window.location.search);
    const typeParam = urlParams.get('type');
    if (typeParam === 'aim' || typeParam === 'cross') {
        currentDataType = typeParam;
    }
    document.getElementById('dtAIO').classList.toggle('primary', currentDataType === 'aio');
    document.getElementById('dtAIM').classList.toggle('primary', currentDataType === 'aim');
    document.getElementById('dtCross').classList.toggle('primary', currentDataType === 'cross');
    document.getElementById('dtAIO').addEventListener('click', () => switchDataType('aio'));
    document.getElementById('dtAIM').addEventListener('click', () => switchDataType('aim'));
    document.getElementById('dtCross').addEventListener('click', () => switchDataType('cross'));
    document.getElementById('gstAIO').addEventListener('click', () => switchGlobalSubTab('aio'));
    document.getElementById('gstAIM').addEventListener('click', () => switchGlobalSubTab('aim'));
    document.getElementById('exportAllBtn').addEventListener('click', async () => {
        await exportAllDataGlobal();
    });
    await setupToggleCaptureButton();
    try {
        const queries = await loadQueries();
        _currentQueries = queries;
        document.getElementById('querySelect').addEventListener('change', async (e) => {
            const selectedIndex = e.target.value;
            if (selectedIndex === '') {
                showData(false);
                return;
            }
            const selectedQuery = _currentQueries[parseInt(selectedIndex)];
            if (selectedQuery) {
                if (currentDataType === 'cross') {
                    const snapshotAI = document.getElementById('snapshotAISection');
                    const venn = document.getElementById('vennSection');
                    if (snapshotAI) snapshotAI.style.display = 'none';
                    if (venn) venn.style.display = 'none';
                    await renderCrossComparison(selectedQuery.query);
                    showData(true);
                } else {
                    await renderQueryDetail(selectedQuery);
                    showData(true);
                }
            }
        });
        document.getElementById('showAllBtn').addEventListener('click', async () => {
            await showAllQueriesView();
        });
        document.getElementById('showDataAttridBtn').addEventListener('click', async () => {
            await showAllDataAttridView();
        });
        document.getElementById('showSingleBtn').addEventListener('click', () => {
            showSingleQueryView();
        });
        const queryParam = urlParams.get('query');
        if (queryParam) {
            const decodedQuery = decodeURIComponent(queryParam);
            const queryIndex = queries.findIndex(q => q.query === decodedQuery);
            if (queryIndex !== -1) {
                const selectElement = document.getElementById('querySelect');
                selectElement.value = queryIndex.toString();
                const selectedQuery = queries[queryIndex];
                if (currentDataType === 'cross') {
                    const snapshotAI = document.getElementById('snapshotAISection');
                    const venn = document.getElementById('vennSection');
                    if (snapshotAI) snapshotAI.style.display = 'none';
                    if (venn) venn.style.display = 'none';
                    await renderCrossComparison(selectedQuery.query);
                } else {
                    await renderQueryDetail(selectedQuery);
                }
                showData(true);
            }
        }
        showLoading(false);
        if (!queryParam || queries.findIndex(q => q.query === decodeURIComponent(queryParam)) === -1) {
            showData(false);
        }
    } catch (error) {
        showLoading(false);
        document.getElementById('noDataContainer').innerHTML = `
            <div class="error">Loading error: ${error.message}</div>
        `;
    }
});
(function() {
    function showTip(icon) {
        const tip = icon.querySelector('.info-tip');
        if (!tip) return;
        tip.style.display = 'block';
        const r = icon.getBoundingClientRect();
        const tw = tip.offsetWidth;
        const th = tip.offsetHeight;
        let left = r.left + r.width / 2 - tw / 2;
        let top = r.top - th - 10;
        if (left < 8) left = 8;
        if (left + tw > window.innerWidth - 8) left = window.innerWidth - tw - 8;
        if (top < 8) { top = r.bottom + 10; tip.classList.add('below'); } else { tip.classList.remove('below'); }
        tip.style.left = left + 'px';
        tip.style.top = top + 'px';
    }
    function hideTip(icon) {
        const tip = icon.querySelector('.info-tip');
        if (tip) tip.style.display = 'none';
    }
    document.addEventListener('mouseover', function(e) {
        const icon = e.target.closest('.info-icon');
        if (!icon) return;
        showTip(icon);
    });
    document.addEventListener('mouseout', function(e) {
        const icon = e.target.closest('.info-icon');
        if (!icon) return;
        if (e.relatedTarget && icon.contains(e.relatedTarget)) return;
        hideTip(icon);
    });
    document.addEventListener('click', function(e) {
        const icon = e.target.closest('.info-icon');
        if (!icon) {
            document.querySelectorAll('.info-tip').forEach(function(t) { t.style.display = 'none'; });
            return;
        }
        const tip = icon.querySelector('.info-tip');
        if (!tip) return;
        if (tip.style.display === 'block') { hideTip(icon); } else { showTip(icon); }
    });
})();
