let allData = null;
let currentTab = 'aio';
const elements = {
  totalQueries: document.getElementById('totalQueries'),
  totalCaptures: document.getElementById('totalCaptures'),
  todayCaptures: document.getElementById('todayCaptures'),
  queriesContainer: document.getElementById('queriesContainer'),
  loading: document.getElementById('loading'),
  emptyState: document.getElementById('emptyState'),
  refreshBtn: document.getElementById('refreshBtn'),
  settingsBtn: document.getElementById('settingsBtn'),
  exportBtn: document.getElementById('exportBtn'),
  viewAllBtn: document.getElementById('viewAllBtn'),
  tabAIO: document.getElementById('tabAIO'),
  tabAIM: document.getElementById('tabAIM')
};
document.addEventListener('DOMContentLoaded', () => {
  loadData();
  setupEventListeners();
  setupToggleCaptureButton();
  setupTabListeners();
});
function setupTabListeners() {
  if (elements.tabAIO) {
    elements.tabAIO.addEventListener('click', () => switchTab('aio'));
  }
  if (elements.tabAIM) {
    elements.tabAIM.addEventListener('click', () => switchTab('aim'));
  }
}
function switchTab(tab) {
  currentTab = tab;
  if (elements.tabAIO) {
    elements.tabAIO.classList.toggle('active', tab === 'aio');
    elements.tabAIO.classList.remove('active-aim');
  }
  if (elements.tabAIM) {
    elements.tabAIM.classList.toggle('active-aim', tab === 'aim');
    elements.tabAIM.classList.toggle('active', false);
    if (tab !== 'aim') elements.tabAIM.classList.remove('active-aim');
  }
  loadData();
}
function setupEventListeners() {
  if (elements.refreshBtn) {
    elements.refreshBtn.addEventListener('click', () => {
      loadData();
    });
  }
  if (elements.settingsBtn) {
    elements.settingsBtn.addEventListener('click', () => {
      try {
        chrome.storage.local.getBytesInUse(null, (bytesInUse) => {
          const mb = (bytesInUse / 1024 / 1024).toFixed(2);
          alert(`Storage used: ${mb} MB\n\nLimit: unlimited\nMax captures: 1000 (AIO + AIM combined)`);
        });
      } catch (error) {
        alert('Error accessing storage');
      }
    });
  }
  if (elements.exportBtn) {
    elements.exportBtn.addEventListener('click', () => {
      showExportMenu();
    });
  }
  if (elements.viewAllBtn) {
    elements.viewAllBtn.addEventListener('click', () => {
      try {
        const viewerUrl = `viewer.html?type=${currentTab}`;
        chrome.runtime.sendMessage({ action: 'openViewer', url: viewerUrl });
        window.close();
      } catch (error) { }
    });
  }
}
function loadData() {
  showLoading(true);
  try {
    chrome.runtime.sendMessage({ action: 'getData' }, (response) => {
      if (chrome.runtime.lastError) {
        allData = { queries: {}, metadata: {} };
      } else {
        allData = convertFlatDataToStructured(response || {}, currentTab);
      }
      updateStats();
      displayQueries();
      showLoading(false);
    });
  } catch (error) {
    allData = { queries: {}, metadata: {} };
    updateStats();
    displayQueries();
    showLoading(false);
  }
}
function convertFlatDataToStructured(flatData, tab) {
  const prefix = tab === 'aim' ? 'aim_' : 'aio_';
  const structured = {
    queries: {},
    metadata: {
      totalCaptures: 0,
      version: '1.0',
      lastUpdate: new Date().toISOString()
    }
  };
  for (const [key, value] of Object.entries(flatData)) {
    if (key.startsWith(prefix) && value && typeof value === 'object') {
      const capture = value;
      const query = capture.searchQuery || 'unknown query';
      if (!structured.queries[query]) {
        structured.queries[query] = {
          captures: [],
          firstSeen: capture.timestamp,
          lastUpdated: capture.timestamp
        };
      }
      structured.queries[query].captures.push(capture);
      if (new Date(capture.timestamp) > new Date(structured.queries[query].lastUpdated)) {
        structured.queries[query].lastUpdated = capture.timestamp;
      }
      structured.metadata.totalCaptures++;
    }
  }
  return structured;
}
function updateStats() {
  if (!allData || !allData.queries) {
    if (elements.totalQueries) elements.totalQueries.textContent = '0';
    if (elements.totalCaptures) elements.totalCaptures.textContent = '0';
    if (elements.todayCaptures) elements.todayCaptures.textContent = '0';
    return;
  }
  const queries = Object.keys(allData.queries);
  let totalCaptures = 0;
  let todayCaptures = 0;
  const today = new Date().toDateString();
  queries.forEach(query => {
    const captures = allData.queries[query].captures || [];
    totalCaptures += captures.length;
    captures.forEach(capture => {
      try {
        const captureDate = new Date(capture.timestamp).toDateString();
        if (captureDate === today) {
          todayCaptures++;
        }
      } catch (e) { }
    });
  });
  if (elements.totalQueries) elements.totalQueries.textContent = queries.length;
  if (elements.totalCaptures) elements.totalCaptures.textContent = totalCaptures;
  if (elements.todayCaptures) elements.todayCaptures.textContent = todayCaptures;
}
function displayQueries() {
  if (!allData || !allData.queries) {
    showEmptyState();
    return;
  }
  const queries = Object.keys(allData.queries);
  if (queries.length === 0) {
    showEmptyState();
    return;
  }
  hideEmptyState();
  const sortedQueries = queries.sort((a, b) => {
    try {
      const dateA = new Date(allData.queries[a].lastUpdated);
      const dateB = new Date(allData.queries[b].lastUpdated);
      return dateB - dateA;
    } catch (e) {
      return 0;
    }
  });
  renderQueries(sortedQueries);
}
function showEmptyState() {
  if (elements.emptyState) elements.emptyState.style.display = 'block';
  if (elements.queriesContainer) elements.queriesContainer.style.display = 'none';
}
function hideEmptyState() {
  if (elements.emptyState) elements.emptyState.style.display = 'none';
  if (elements.queriesContainer) elements.queriesContainer.style.display = 'block';
}
function renderQueries(queries) {
  if (!elements.queriesContainer) {
    return;
  }
  elements.queriesContainer.innerHTML = '';
  if (queries.length === 0) {
    elements.queriesContainer.innerHTML = `
      <div class="empty-state">
        <p>No query found</p>
      </div>
    `;
    return;
  }
  queries.forEach(query => {
    try {
      const queryData = allData.queries[query];
      const captures = queryData.captures || [];
      const citationCount = captures.reduce((sum, c) => sum + (c.citations || []).length, 0);
      const lastUpdate = queryData.lastUpdated ? new Date(queryData.lastUpdated) : new Date();
      let statusText;
      if (currentTab === 'aim') {
        const aimStatus = captures.length > 0 ? (captures[0].aimStatus || 'unknown') : 'unknown';
        statusText = aimStatus;
      } else {
        const aioStatus = captures.length > 0 ? (captures[0].aioStatus || 'unknown') : 'unknown';
        statusText = aioStatus;
      }
      const queryElement = createQueryElement(query, citationCount, lastUpdate, statusText);
      elements.queriesContainer.appendChild(queryElement);
    } catch (error) { }
  });
}
function createQueryElement(query, citationCount, lastUpdate, statusText) {
  const div = document.createElement('div');
  div.className = 'query-item';
  const timeAgo = getTimeAgo(lastUpdate);
  const badgeClass = currentTab === 'aim' ? 'type-badge-aim' : 'type-badge-aio';
  const badgeLabel = currentTab === 'aim' ? 'AIM' : 'AIO';
  const statusBadge = statusText === 'displayed' ? badgeLabel : (statusText === 'magi_only' ? 'Grounding only' : statusText);
  div.innerHTML = `
    <div class="query-text">${escapeHtml(query)} <span class="type-badge ${badgeClass}">${badgeLabel}</span></div>
    <div class="query-meta">
      <span>${timeAgo}</span>
      <span class="query-count">${citationCount} citation${citationCount !== 1 ? 's' : ''} · ${statusBadge}</span>
    </div>
  `;
  div.addEventListener('click', () => {
    try {
      const encodedQuery = encodeURIComponent(query);
      const viewerUrl = `viewer.html?query=${encodedQuery}&type=${currentTab}`;
      chrome.runtime.sendMessage({
        action: 'openViewer',
        url: viewerUrl
      });
      window.close();
    } catch (error) { }
  });
  return div;
}
function showExportMenu() {
  const menu = document.createElement('div');
  menu.style.cssText = `
    position: fixed;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    background: white;
    padding: 20px;
    border-radius: 8px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.15);
    z-index: 1000;
  `;
  menu.innerHTML = `
    <h3 style="margin-bottom: 15px; color: #202124;">Export ${currentTab.toUpperCase()} data</h3>
    <button class="button button-primary" style="display: block; width: 200px; margin-bottom: 10px;" id="exportJson">
      Full JSON export
    </button>
    <button class="button button-secondary" style="display: block; width: 200px; margin-bottom: 10px;" id="exportCsv">
      Simplified CSV export
    </button>
    <button class="button button-secondary" style="display: block; width: 200px;" id="cancelExport">
      Cancel
    </button>
  `;
  document.body.appendChild(menu);
  const overlay = document.createElement('div');
  overlay.style.cssText = `
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0,0,0,0.5);
    z-index: 999;
  `;
  document.body.appendChild(overlay);
  const exportJsonBtn = document.getElementById('exportJson');
  const exportCsvBtn = document.getElementById('exportCsv');
  const cancelBtn = document.getElementById('cancelExport');
  if (exportJsonBtn) {
    exportJsonBtn.addEventListener('click', () => {
      exportData('json');
      closeExportMenu();
    });
  }
  if (exportCsvBtn) {
    exportCsvBtn.addEventListener('click', () => {
      exportData('csv');
      closeExportMenu();
    });
  }
  if (cancelBtn) {
    cancelBtn.addEventListener('click', closeExportMenu);
  }
  overlay.addEventListener('click', closeExportMenu);
  function closeExportMenu() {
    if (menu.parentNode) menu.parentNode.removeChild(menu);
    if (overlay.parentNode) overlay.parentNode.removeChild(overlay);
  }
}
function exportData(format) {
  try {
    if (!allData || !allData.queries) {
      alert('No data to export');
      return;
    }
    const captures = [];
    for (const queryData of Object.values(allData.queries)) {
      for (const capture of (queryData.captures || [])) {
        captures.push({ ...capture, type: currentTab });
      }
    }
    let output;
    let mimeType;
    let extension;
    if (format === 'json') {
      function cleanObj(obj) {
        if (Array.isArray(obj)) return obj.map(cleanObj);
        if (obj && typeof obj === 'object') {
          const out = {};
          for (const [k, v] of Object.entries(obj)) {
            if (v === null || v === undefined) continue;
            if (v === '' || v === 0 || v === false) continue;
            if (Array.isArray(v) && v.length === 0) continue;
            out[k] = cleanObj(v);
          }
          return out;
        }
        return obj;
      }
      const cleanCaptures = captures.map(c => {
        const clean = { ...c };
        delete clean.fanOutQuery;
        delete clean.followUpQuestions;
        delete clean.hasAboutResult;
        delete clean.additionalTextFragments;
        delete clean.citationButtonUUIDs;
        delete clean.captureType;
        delete clean.type;
        if (clean.magiInnerHTML) {
          clean.GroundingURLs = clean.magiInnerHTML;
          delete clean.magiInnerHTML;
        }
        if (clean.imageCarouselSources) {
          clean.ImageCarouselSources = clean.imageCarouselSources;
          delete clean.imageCarouselSources;
        }
        if (clean.citations) {
          clean.citations = clean.citations.map(cit => {
            const cc = { ...cit };
            delete cc.snippet;
            delete cc.siteName;
            delete cc.uuid;
            delete cc.publishedDate;
            return cc;
          });
        }
        if (clean.GroundingURLs) {
          clean.GroundingURLs = clean.GroundingURLs.map(g => {
            const gg = { ...g };
            delete gg.hasAboutResult;
            delete gg.hasReqBlob;
            delete gg.sourceLabel;
            delete gg.count;
            return gg;
          });
        }
        for (const field of ['sidebarCards', 'sidebarSources', 'textFragmentLinks']) {
          if (clean[field]) {
            clean[field] = clean[field].map(s => {
              const ss = { ...s };
              delete ss.snippet;
              delete ss.uuid;
              delete ss.siteName;
              return ss;
            });
          }
        }
        delete clean.ffdbKGIds;
        return cleanObj(clean);
      });
      const exportPayload = {
        type: currentTab,
        captures: cleanCaptures,
        exportDate: new Date().toISOString(),
        totalItems: cleanCaptures.length
      };
      output = JSON.stringify(exportPayload, null, 2);
      mimeType = 'application/json';
      extension = 'json';
    } else {
      const statusField = currentTab === 'aim' ? 'AIM Status' : 'AIO Status';
      let csv = `Query,${statusField},Citations,Generated Text Length,Timestamp,PageUrl\n`;
      captures.forEach(c => {
        const query = (c.searchQuery || '').replace(/"/g, '""');
        const status = currentTab === 'aim' ? (c.aimStatus || '') : (c.aioStatus || '');
        const citCount = (c.citations || []).length;
        const textLen = (c.generatedText || '').length;
        const ts = c.timestamp || '';
        const pageUrl = (c.pageUrl || '').replace(/"/g, '""');
        csv += `"${query}","${status}","${citCount}","${textLen}","${ts}","${pageUrl}"\n`;
      });
      output = csv;
      mimeType = 'text/csv';
      extension = 'csv';
    }
    const blob = new Blob([output], { type: mimeType });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `${currentTab}-inspector-${new Date().toISOString().split('T')[0]}.${extension}`;
    a.click();
    URL.revokeObjectURL(url);
    showNotification(`${format.toUpperCase()} export successful!`);
  } catch (error) {
    alert('Export error: ' + error.message);
  }
}
function showNotification(message) {
  const notif = document.createElement('div');
  notif.style.cssText = `
    position: fixed;
    bottom: 20px;
    left: 50%;
    transform: translateX(-50%);
    background: #1a73e8;
    color: white;
    padding: 12px 24px;
    border-radius: 4px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.2);
    z-index: 1001;
  `;
  notif.textContent = message;
  document.body.appendChild(notif);
  setTimeout(() => {
    notif.style.opacity = '0';
    notif.style.transition = 'opacity 0.3s';
    setTimeout(() => {
      if (notif.parentNode) notif.parentNode.removeChild(notif);
    }, 300);
  }, 3000);
}
function showLoading(show) {
  if (elements.loading) {
    elements.loading.style.display = show ? 'block' : 'none';
  }
  if (elements.queriesContainer) {
    elements.queriesContainer.style.display = show ? 'none' : 'block';
  }
  if (elements.emptyState) {
    elements.emptyState.style.display = 'none';
  }
}
function escapeHtml(unsafe) {
  if (typeof unsafe !== 'string') return '';
  return unsafe
    .replace(/&/g, "&amp;")
    .replace(/</g, "&lt;")
    .replace(/>/g, "&gt;")
    .replace(/"/g, "&quot;")
    .replace(/'/g, "&#039;");
}
function getTimeAgo(date) {
  if (!date) return 'Unknown date';
  try {
    const seconds = Math.floor((new Date() - date) / 1000);
    if (seconds < 60) return 'Just now';
    if (seconds < 3600) return `${Math.floor(seconds / 60)} min ago`;
    if (seconds < 86400) return `${Math.floor(seconds / 3600)} h ago`;
    if (seconds < 604800) return `${Math.floor(seconds / 86400)} d ago`;
    return date.toLocaleDateString('en-US');
  } catch (error) {
    return 'Invalid date';
  }
}
function setupToggleCaptureButton() {
  const toggleBtn = document.getElementById('toggleCaptureBtn');
  chrome.storage.local.get(['captureEnabled'], (result) => {
    const isEnabled = result.captureEnabled !== false;
    updateToggleButton(toggleBtn, isEnabled);
  });
  toggleBtn.addEventListener('click', () => {
    chrome.storage.local.get(['captureEnabled'], (result) => {
      const currentState = result.captureEnabled !== false;
      const newState = !currentState;
      chrome.storage.local.set({ captureEnabled: newState }, () => {
        updateToggleButton(toggleBtn, newState);
        showNotification(newState ? 'Capture enabled' : 'Capture paused');
      });
    });
  });
}
function updateToggleButton(button, isEnabled) {
  if (isEnabled) {
    button.innerHTML = 'Pause';
    button.style.background = '#f59e0b';
    button.style.color = 'white';
  } else {
    button.innerHTML = 'Enable';
    button.style.background = '#10b981';
    button.style.color = 'white';
  }
}
