const MAX_STORAGE_ITEMS = 1000;
function _mergeFields(target, source, fields) {
  let changed = false;
  for (const f of fields) {
    const tVal = target[f], sVal = source[f];
    const tEmpty = !tVal || (Array.isArray(tVal) && tVal.length === 0);
    const sEmpty = !sVal || (Array.isArray(sVal) && sVal.length === 0);
    if (tEmpty && !sEmpty) { target[f] = sVal; changed = true; }
  }
  return changed;
}
async function _handleCapture(data, type) {
  const isAIO = type === 'aio';
  const prefix = isAIO ? 'aio_' : 'aim_';
  const statusField = isAIO ? 'aioStatus' : 'aimStatus';
  const sidebarField = isAIO ? 'sidebarCards' : 'sidebarSources';
  const preserveFields = isAIO
    ? ['entityLinks', 'externalLinks', 'additionalTextFragments', 'followUpQuestions', 'organicUrls', 'imageCarouselSources', 'textFragmentLinks']
    : ['placeEntities', 'kgEntities', 'ffdbKGIds', 'textFragmentLinks', 'entityLinks', 'externalLinks', 'imageCarouselSources'];
  try {
    const settings = await chrome.storage.local.get(['captureEnabled']);
    if (settings.captureEnabled === false) return;
    const searchQuery = data.searchQuery || '';
    if (!searchQuery) return;
    const citations = data.citations || [];
    const generatedText = data.generatedText || '';
    const status = data[statusField] || 'absent';
    const sidebarItems = data[sidebarField] || [];
    const magiInnerHTML = data.magiInnerHTML || [];
    if (citations.length === 0 && !generatedText && status === 'absent' && sidebarItems.length === 0 && magiInnerHTML.length === 0) return;
    const existingData = await chrome.storage.local.get(null);
    let existingKey = null;
    let existingValue = null;
    for (const [k, v] of Object.entries(existingData)) {
      if (k.startsWith(prefix) && typeof v === 'object' && v.searchQuery === searchQuery) {
        existingKey = k;
        existingValue = v;
        break;
      }
    }
    if (existingKey && existingValue) {
      const existingHasText = !!(existingValue.generatedText);
      const existingCitCount = (existingValue.citations || []).length;
      const existingSidebarCount = (existingValue[sidebarField] || []).length;
      const existingMagiCount = (existingValue.magiInnerHTML || []).length;
      const newHasText = !!generatedText;
      const newCitCount = citations.length;
      const newSidebarCount = sidebarItems.length;
      const newMagiCount = magiInnerHTML.length;
      const shouldReplace = (newHasText && !existingHasText) || newCitCount > existingCitCount ||
        newSidebarCount > existingSidebarCount || newMagiCount > existingMagiCount ||
        (isAIO && data.fanOutQuery && !existingValue.fanOutQuery);
      if (shouldReplace) {
        _mergeFields(data, existingValue, preserveFields);
        await chrome.storage.local.remove(existingKey);
      } else {
        const needsUpdate = _mergeFields(existingValue, data, preserveFields);
        if (needsUpdate) {
          await chrome.storage.local.set({ [existingKey]: existingValue });
        }
        return;
      }
    }
    const entry = {
      searchQuery, citations, generatedText,
      [statusField]: status,
      magiInnerHTML,
      entityLinks: data.entityLinks || [],
      externalLinks: data.externalLinks || [],
      imageCarouselSources: data.imageCarouselSources || [],
      timestamp: data.timestamp || new Date().toISOString(),
      pageUrl: data.pageUrl || ''
    };
    if (isAIO) {
      entry.fanOutQuery = data.fanOutQuery || null;
      entry.followUpQuestions = data.followUpQuestions || null;
      entry.additionalTextFragments = data.additionalTextFragments || [];
      entry.sidebarCards = data.sidebarCards || [];
      entry.citationButtonUUIDs = data.citationButtonUUIDs || [];
      entry.organicUrls = data.organicUrls || [];
      entry.textFragmentLinks = data.textFragmentLinks || [];
    } else {
      entry.sidebarSources = sidebarItems;
      entry.placeEntities = data.placeEntities || [];
      entry.kgEntities = data.kgEntities || [];
      entry.ffdbKGIds = data.ffdbKGIds || [];
      entry.textFragmentLinks = data.textFragmentLinks || [];
    }
    const key = `${prefix}${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
    await chrome.storage.local.set({ [key]: entry });
    await checkStorageLimit();
  } catch (error) {
    throw error;
  }
}
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'getData') {
    chrome.storage.local.get(null, (data) => {
      if (chrome.runtime.lastError) {
        sendResponse({});
      } else {
        sendResponse(data || {});
      }
    });
    return true;
  }
  if (request.action === 'openViewer') {
    const viewerUrl = request.url ? chrome.runtime.getURL(request.url) : chrome.runtime.getURL('viewer.html');
    chrome.tabs.create({ url: viewerUrl });
  }
  if (request.action === 'captureAIO') {
    _handleCapture(request.data, 'aio')
      .then(() => sendResponse({ success: true }))
      .catch((error) => sendResponse({ success: false, error: error.message }));
    return true;
  }
  if (request.action === 'captureAIM') {
    _handleCapture(request.data, 'aim')
      .then(() => sendResponse({ success: true }))
      .catch((error) => sendResponse({ success: false, error: error.message }));
    return true;
  }
  if (request.action === 'clearData') {
    clearData(request.query, request.type || null)
      .then(() => sendResponse({ success: true }))
      .catch((error) => sendResponse({ success: false, error: error.message }));
    return true;
  }
});
async function checkStorageLimit() {
  try {
    const data = await chrome.storage.local.get(null);
    const items = Object.entries(data).filter(([key, value]) =>
      (key.startsWith('aio_') || key.startsWith('aim_')) && typeof value === 'object'
    );
    if (items.length > MAX_STORAGE_ITEMS) {
      items.sort(([, a], [, b]) => {
        const timeA = new Date(a.timestamp || 0).getTime();
        const timeB = new Date(b.timestamp || 0).getTime();
        return timeA - timeB;
      });
      const toRemove = items.length - MAX_STORAGE_ITEMS;
      const keysToRemove = items.slice(0, toRemove).map(([key]) => key);
      await chrome.storage.local.remove(keysToRemove);
    }
  } catch (error) { }
}
async function clearData(query = null, type = null) {
  try {
    const data = await chrome.storage.local.get(null);
    const prefixes = type === 'aio' ? ['aio_'] : type === 'aim' ? ['aim_'] : ['aio_', 'aim_'];
    if (query) {
      const keysToRemove = Object.entries(data)
        .filter(([key, value]) =>
          prefixes.some(p => key.startsWith(p)) &&
          typeof value === 'object' &&
          value.searchQuery === query
        )
        .map(([key]) => key);
      await chrome.storage.local.remove(keysToRemove);
    } else {
      const keysToRemove = Object.keys(data).filter(key => prefixes.some(p => key.startsWith(p)));
      if (keysToRemove.length > 0) {
        await chrome.storage.local.remove(keysToRemove);
      }
    }
  } catch (error) {
    throw error;
  }
}
